package com.el.coordinator.file.api;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.BooleanUtil;
import cn.hutool.core.util.ObjectUtil;
import com.el.coordinator.core.common.api.ApiResult;
import com.el.coordinator.core.common.jpa.vo.PagingVO;
import com.el.coordinator.file.enums.FsmApiConstantEnum;
import com.el.coordinator.file.parameter.CreatFileUploadInfoParam;
import com.el.coordinator.file.parameter.FilePackageParam;
import com.el.coordinator.file.parameter.FileUploadInfoQueryParam;
import com.el.coordinator.file.parameter.StartFileUploadParam;
import com.el.coordinator.file.service.FileServiceApiInterface;
import com.el.coordinator.file.utils.ParserUtil;
import com.el.coordinator.file.vo.*;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.http.*;
import org.springframework.lang.Nullable;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.util.StreamUtils;
import org.springframework.web.client.RequestCallback;
import org.springframework.web.client.ResponseExtractor;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import java.io.File;
import java.io.OutputStream;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * <p>
 * 文件管理服务的调用工具类
 * </p >
 *
 * @author niu.chen
 * @date 2021/2/13
 */
@Slf4j
public class FsmApiRequest implements FileServiceApiInterface {
    private final String serviceUrl;
    private final RestTemplate restTemplate;
    private final ObjectMapper objectMapper;
    private final DateTimeFormatter formatterLong = DateTimeFormatter.ofPattern("yyyyMMddHHmmssSSS");

    public FsmApiRequest(String serviceUri, RestTemplate restTemplate, ObjectMapper objectMapper) {
        this.serviceUrl = serviceUri;
        this.restTemplate = restTemplate;
        this.objectMapper = objectMapper;
    }

    @Override
    public ApiResult<CreatFileUploadInfoVO> applyFileUploadInfo(CreatFileUploadInfoParam creatFileUploadInfoParam) {
        String url = serviceUrl +
                FsmApiConstant.DEFAULT_FSM_CONTEXT_PATH +
                FsmApiConstant.DEFAULT_FSM_SERVICE_API +
                FsmApiConstant.API_POST_UPLOAD_APPLY;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        try {
            var result = restTemplate.exchange(url, HttpMethod.POST, new HttpEntity<>(creatFileUploadInfoParam, headers), String.class);
            return convertResult(result, new TypeReference<>() {
            }, url, "上传文件失败，请确认文件服务配置正常");
        } catch (Exception e) {
            log.error("上传文件异常：{}", url, e);
            return ApiResult.fail("上传文件失败：" + e.getMessage());
        }
    }

    @Override
    public ApiResult<StartFileUploadVO> startFileUploadFile(File file, StartFileUploadParam startFileUploadParam) {
        FileSystemResource resource = new FileSystemResource(file);
        return startFileUploadResource(resource, startFileUploadParam);
    }

    @Override
    public ApiResult<StartFileUploadVO> startFileUploadResource(Resource resource, StartFileUploadParam startFileUploadParam) {
        String url = serviceUrl +
                FsmApiConstant.DEFAULT_FSM_CONTEXT_PATH +
                FsmApiConstant.DEFAULT_FSM_SERVICE_API +
                FsmApiConstantEnum.API_POST_UPLOAD.getUrl();
        MultiValueMap<String, Object> requestEntity = new LinkedMultiValueMap<>();
        requestEntity.add("fileCode", startFileUploadParam.getFileCode());
        requestEntity.add("fileToKen", startFileUploadParam.getFileToKen());
        requestEntity.add("file", resource);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.MULTIPART_FORM_DATA);

        try {
            var result = restTemplate.exchange(url, HttpMethod.POST, new HttpEntity<>(requestEntity, headers), String.class);
            return convertResult(result, new TypeReference<>() {
            }, url, "上传文件失败，请确认文件服务配置正常");
        } catch (Exception e) {
            log.error("上传文件异常：{}", url, e);
            return ApiResult.fail("上传文件失败：" + e.getMessage());
        }
    }

    @Override
    public ApiResult<Long> applyChunk(CreatFileUploadInfoParam reqVO) {
        String url = serviceUrl +
                FsmApiConstant.DEFAULT_FSM_CONTEXT_PATH +
                FsmApiConstant.DEFAULT_FSM_SERVICE_API +
                FsmApiConstant.API_CHUNK_APPLY;

        try {
            var result = restTemplate.exchange(url, HttpMethod.POST, new HttpEntity<>(reqVO, null), String.class);
            return convertResult(result, new TypeReference<>() {
            }, url, "上传文件失败，请确认文件服务配置正常");
        } catch (Exception e) {
            log.error("申请分片上传文件异常：{}", url, e);
            return ApiResult.fail("上传文件失败：" + e.getMessage());
        }
    }

    @Override
    public ApiResult<StartFileUploadVO> uploadChunk(@Nullable Resource file, FileChunkSaveVO saveVO) {
        String url = serviceUrl +
                FsmApiConstant.DEFAULT_FSM_CONTEXT_PATH +
                FsmApiConstant.DEFAULT_FSM_SERVICE_API +
                FsmApiConstant.API_CHUNK_UPLOAD;
        // 请求体
        MultiValueMap<String, Object> requestEntity = new LinkedMultiValueMap<>();
        if (file != null) {
            requestEntity.add("file", file);
        }
        requestEntity.add("uploadId", saveVO.getUploadId());
        requestEntity.add("merge", Boolean.TRUE.equals(saveVO.getMerge()));
        if (saveVO.getPartNumber() != null) {
            requestEntity.add("partNumber", saveVO.getPartNumber());
        }

        // 请求头
        HttpHeaders headers = new HttpHeaders();
        headers.set(HttpHeaders.CONTENT_TYPE, MediaType.MULTIPART_FORM_DATA_VALUE);

        try {
            var result = restTemplate.exchange(url, HttpMethod.POST, new HttpEntity<>(requestEntity, headers), String.class);
            return convertResult(result, new TypeReference<>() {
            }, url, "上传文件失败，请确认文件服务配置正常");
        } catch (Exception e) {
            log.error("分片上传异常：{}", url, e);
            return ApiResult.fail("上传文件失败：" + e.getMessage());
        }
    }


    @Override
    public ApiResult<Integer> deleteFile(String fileCode) {
        String url = serviceUrl +
                FsmApiConstant.DEFAULT_FSM_CONTEXT_PATH +
                FsmApiConstant.DEFAULT_FSM_SERVICE_API +
                FsmApiConstant.API_PUT_DELETE_FILE;
        url = ParserUtil.parse1(url, fileCode);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);

        try {
            var result = restTemplate.exchange(url, HttpMethod.PUT, new HttpEntity<>(null, headers), String.class);
            return convertResult(result, new TypeReference<>() {
            }, url, "删除文件失败，请确认文件服务配置正常");
        } catch (Exception e) {
            log.error("删除文件异常：{}", url, e);
            return ApiResult.fail("删除文件失败：" + e.getMessage());
        }
    }


    @Override
    public ApiResult<PagingVO<FileUploadInfoVO>> search(FileUploadInfoQueryParam fileUploadInfoQueryParam) {
        String url = serviceUrl +
                FsmApiConstant.DEFAULT_FSM_CONTEXT_PATH +
                FsmApiConstant.DEFAULT_FSM_SERVICE_API +
                FsmApiConstant.API_POST_SEARCH;

        HttpHeaders headers = new HttpHeaders();

        try {
            var result = restTemplate.exchange(url, HttpMethod.POST, new HttpEntity<>(fileUploadInfoQueryParam, headers), String.class);
            return convertResult(result, new TypeReference<>() {
            }, url, "查询文件失败，请确认文件服务配置正常");
        } catch (Exception e) {
            log.error("查询文件异常：{}", url, e);
            return ApiResult.fail("查询文件失败：" + e.getMessage());
        }
    }


    @Override
    public ApiResult<FileUploadInfoVO> findFileCodeOne(String fileCode) {
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append(serviceUrl)
                .append(FsmApiConstant.DEFAULT_FSM_CONTEXT_PATH)
                .append(FsmApiConstant.DEFAULT_FSM_SERVICE_API)
                .append(FsmApiConstant.API_GET_File_CODE);

        String url = stringBuilder.toString();
        url = ParserUtil.parse1(url, fileCode);

        ParameterizedTypeReference<ApiResult<FileUploadInfoVO>> typeRef =
                new ParameterizedTypeReference<>() {
                };
        HttpHeaders headers = new HttpHeaders();

        try {
            var result = restTemplate.exchange(url, HttpMethod.GET, new HttpEntity<>(null, headers), String.class);
            return convertResult(result, new TypeReference<>() {
            }, url, "查询文件失败，请确认文件服务配置正常");
        } catch (Exception e) {
            log.error("查询文件异常：{}", url, e);
            return ApiResult.fail("查询文件失败：" + e.getMessage());
        }
    }

    @Override
    public ApiResult<Boolean> exsits(String fileCode) {
        String url = serviceUrl +
                FsmApiConstant.DEFAULT_FSM_CONTEXT_PATH +
                FsmApiConstant.DEFAULT_FSM_SERVICE_API +
                FsmApiConstant.API_GET_FILE_EXISTS;
        url = ParserUtil.parse1(url, fileCode);

        ParameterizedTypeReference<ApiResult<Boolean>> typeRef =
                new ParameterizedTypeReference<>() {
                };
        HttpHeaders headers = new HttpHeaders();
        try {
            var result = restTemplate.exchange(url, HttpMethod.GET, new HttpEntity<>(null, headers), String.class);
            return convertResult(result, new TypeReference<>() {
            }, url, "查询文件失败，请确认文件服务配置正常");
        } catch (Exception e) {
            log.error("查询文件异常：{}", url, e);
            return ApiResult.fail("查询文件失败：" + e.getMessage());
        }
    }

    @Override
    public ApiResult<Boolean> exsits(List<String> fileCodes) {
        String url = serviceUrl +
                FsmApiConstant.DEFAULT_FSM_CONTEXT_PATH +
                FsmApiConstant.DEFAULT_FSM_SERVICE_API +
                FsmApiConstant.API_POST_FILE_EXISTS;

        ParameterizedTypeReference<ApiResult<Boolean>> typeRef =
                new ParameterizedTypeReference<>() {
                };
        HttpHeaders headers = new HttpHeaders();
        try {
            var result = restTemplate.exchange(url, HttpMethod.POST, new HttpEntity<>(fileCodes, headers), String.class);
            return convertResult(result, new TypeReference<>() {
            }, url, "查询文件失败，请确认文件服务配置正常");
        } catch (Exception e) {
            log.error("查询文件异常：{}", url, e);
            return ApiResult.fail("查询文件失败：" + e.getMessage());
        }
    }

    @Override
    public HttpEntity<Resource> download(String fileCode, String childFlag) {
        String url = serviceUrl + FsmApiConstant.DEFAULT_FSM_CONTEXT_PATH + FsmApiConstant.DEFAULT_FSM_SERVICE_API
                + FsmApiConstant.API_GET_DOWNLOAD + "?childFlag=" + ObjectUtil.defaultIfNull(childFlag, "");

        ResponseEntity<byte[]> entity = null;
        MediaType mediaType = null;
        int times = 1;
        while (true) {
            try {
                entity = restTemplate.exchange(url, HttpMethod.GET, null, byte[].class, fileCode, childFlag);
            } catch (RestClientException e) {
                log.error("下载文件异常：{}", fileCode, e);
            }

            if (entity != null && entity.getStatusCode().is2xxSuccessful() && ArrayUtil.isNotEmpty(entity.getBody())) {
                // 下载成功
                mediaType = entity.getHeaders().getContentType();
                break;
            }

            if (times >= 3) {
                break;
            }
            times++;
            try {
                TimeUnit.SECONDS.sleep(times);
            } catch (InterruptedException e) {
            }
        }

        if (entity == null) {
            // 下载失败
            return ResponseEntity.badRequest().build();
        }
        if (ArrayUtil.isEmpty(entity.getBody())) {
            return ResponseEntity.notFound().build();
        }

        return ResponseEntity.ok()
                .contentType(ObjectUtil.defaultIfNull(mediaType, MediaType.APPLICATION_OCTET_STREAM))
                .body(new ByteArrayResource(entity.getBody()))
                ;
    }

    @Override
    public HttpEntity<StreamingResponseBody> downloadStreaming(String fileCode, String childFlag) {
        FileUploadInfoVO infoVO = findFileCodeOne(fileCode).getData();
        int times = 1;
        while (infoVO == null && times < 5) {
            try {
                TimeUnit.SECONDS.sleep(times);
            } catch (InterruptedException e) {
            }
            findFileCodeOne(fileCode).getData();
            times++;
        }
        if (infoVO == null) {
            return ResponseEntity.notFound().build();
        }

        ContentDisposition contentDisposition = ContentDisposition.builder("attachment")
                .filename(URLEncoder.encode(infoVO.getOriginalName(), StandardCharsets.UTF_8))
                .build();

        StreamingResponseBody responseBody = outputStream -> {
            writeStream(fileCode, childFlag, outputStream);
            outputStream.close();
        };

        return ResponseEntity.ok()
                .contentType(MediaType.APPLICATION_OCTET_STREAM)
                .header(HttpHeaders.CONTENT_DISPOSITION, contentDisposition.toString())
                .body(responseBody);
    }

    @Override
    public ResponseEntity<StreamingResponseBody> downloadByPackage(FilePackageParam param) {
        String url = serviceUrl + FsmApiConstant.DEFAULT_FSM_CONTEXT_PATH + FsmApiConstant.DEFAULT_FSM_SERVICE_API
                + FsmApiConstant.API_PACKAGE_DOWNLOAD;
        RequestCallback requestCallback = restTemplate.httpEntityCallback(param);

        ContentDisposition contentDisposition = ContentDisposition.builder("attachment")
                .filename(URLEncoder.encode(CharSequenceUtil.blankToDefault(param.getPackageName(), formatterLong.format(LocalDateTime.now())) + ".zip", StandardCharsets.UTF_8))
                .build();

        StreamingResponseBody responseBody = outputStream -> {
            ResponseExtractor<Boolean> responseExtractor = clientHttpResponse -> {
                if (clientHttpResponse.getStatusCode().is2xxSuccessful()) {
                    StreamUtils.copy(clientHttpResponse.getBody(), outputStream);
                    return true;
                }
                return false;
            };
            var result = restTemplate.execute(url, HttpMethod.POST, requestCallback, responseExtractor);
            log.info("下载结果：{}，{}", url, result);
        };

        return ResponseEntity.ok()
                .contentType(new MediaType("application", "zip"))
                .header(HttpHeaders.CONTENT_DISPOSITION, contentDisposition.toString())
                .body(responseBody);
    }

    @Override
    public ApiResult<Long> applyPackage(FilePackageParam param) {
        String url = serviceUrl +
                FsmApiConstant.DEFAULT_FSM_CONTEXT_PATH +
                FsmApiConstant.DEFAULT_FSM_SERVICE_API +
                FsmApiConstant.API_PACKAGE_APPLY;
        try {
            var result = restTemplate.exchange(url, HttpMethod.POST, new HttpEntity<>(param, null), String.class);
            return convertResult(result, new TypeReference<>() {
            }, url, "打包下载文件失败，请确认文件服务配置正常");
        } catch (Exception e) {
            log.error("打包下载文件异常：{}", url, e);
            return ApiResult.fail("打包下载文件失败：" + e.getMessage());
        }
    }

    @Override
    public ApiResult<PackageResultVO> packageResult(Long applyId) {
        String url = serviceUrl +
                FsmApiConstant.DEFAULT_FSM_CONTEXT_PATH +
                FsmApiConstant.DEFAULT_FSM_SERVICE_API +
                FsmApiConstant.API_PACKAGE_FINISHED + "?applyId=" + applyId;

        try {
            var result = restTemplate.exchange(url, HttpMethod.GET, null, String.class);
            return convertResult(result, new TypeReference<>() {
            }, url, "打包下载文件失败，请确认文件服务配置正常");
        } catch (Exception e) {
            log.error("打包下载文件异常：{}", url, e);
            return ApiResult.fail("打包下载文件失败：" + e.getMessage());
        }
    }

    @Override
    public ResponseEntity<StreamingResponseBody> downloadPackage(Long applyId, String packageName) {
        String url = serviceUrl + FsmApiConstant.DEFAULT_FSM_CONTEXT_PATH + FsmApiConstant.DEFAULT_FSM_SERVICE_API
                + FsmApiConstant.API_PACKAGE_APPLY_DOWNLOAD + "?applyId=" + applyId;

        ContentDisposition contentDisposition = ContentDisposition.builder("attachment")
                .filename(URLEncoder.encode(CharSequenceUtil.blankToDefault(packageName, formatterLong.format(LocalDateTime.now())) + ".zip", StandardCharsets.UTF_8))
                .build();

        StreamingResponseBody responseBody = outputStream -> {
            ResponseExtractor<Boolean> responseExtractor = clientHttpResponse -> {
                if (clientHttpResponse.getStatusCode().is2xxSuccessful()) {
                    StreamUtils.copy(clientHttpResponse.getBody(), outputStream);
                    return true;
                }
                return false;
            };
            var result = restTemplate.execute(url, HttpMethod.GET, null, responseExtractor);
            log.info("打包下载结果：{}，{}", url, result);
        };

        return ResponseEntity.ok()
                .contentType(new MediaType("application", "zip"))
                .header(HttpHeaders.CONTENT_DISPOSITION, contentDisposition.toString())
                .body(responseBody);
    }

    private <T> T convertResult(ResponseEntity<String> resp, TypeReference<T> resultTypeRef,
                                String url, String failMsg) throws JsonProcessingException {
        if (resp.getStatusCode().is2xxSuccessful() && resp.getBody() != null) {
            return objectMapper.readValue(resp.getBody(), resultTypeRef);
        }
        log.error("文件服务异常：{}， {}, {}", url, resp.getStatusCodeValue(), resp.getBody());
        return (T) ApiResult.fail(failMsg);
    }

    private void writeStream(String fileCode, String childFlag, OutputStream outputStream) {
        String url = serviceUrl + FsmApiConstant.DEFAULT_FSM_CONTEXT_PATH + FsmApiConstant.DEFAULT_FSM_SERVICE_API
                + FsmApiConstant.API_GET_DOWNLOAD + "?childFlag=" + ObjectUtil.defaultIfNull(childFlag, "");

        RequestCallback requestCallback = request -> request.getHeaders().setAccept(List.of(MediaType.APPLICATION_OCTET_STREAM, MediaType.ALL));

        ResponseExtractor<Boolean> responseExtractor = clientHttpResponse -> {
            if (clientHttpResponse.getStatusCode().is2xxSuccessful()) {
                StreamUtils.copy(clientHttpResponse.getBody(), outputStream);
                return true;
            }
            return false;
        };

        Boolean result = false;
        int times = 1;
        while (true) {
            try {
                result = restTemplate.execute(url, HttpMethod.GET, requestCallback, responseExtractor, fileCode);
            } catch (Exception e) {
                log.error("下载文件失败:", e);
            }

            if (BooleanUtil.isTrue(result)) {
                // 下载成功
                break;
            }

            if (times >= 3) {
                break;
            }
            times++;
            try {
                TimeUnit.SECONDS.sleep(times);
            } catch (InterruptedException e) {
            }
        }
    }
}
