package com.el.coordinator.boot.fsm.controller;

import cn.hutool.core.util.ArrayUtil;
import cn.hutool.extra.servlet.ServletUtil;
import com.el.coordinator.boot.fsm.common.ConstantsFile;
import com.el.coordinator.boot.fsm.config.FileConfigProperties;
import com.el.coordinator.boot.fsm.model.vo.FileObjRespVO;
import com.el.coordinator.boot.fsm.service.FileService;
import com.el.coordinator.core.common.api.ApiResult;
import com.el.coordinator.file.parameter.FilePackageParam;
import com.el.coordinator.file.vo.FileChunkReqVO;
import com.el.coordinator.file.vo.FileChunkSaveVO;
import com.el.coordinator.file.vo.PackageResultVO;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.github.xiaoymin.knife4j.annotations.ApiSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.http.*;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;

/**
 * 文件管理
 *
 * @author Kaiser（wang shao）
 * @date 2021-04-18
 */
@Api(tags = "文件服务")
@ApiSupport(author = "wang shao", order = 1)
@RestController
@RequestMapping(value = "/com/file/v1", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
@SuppressWarnings("unchecked")
public class FileController {

    private static final Logger log = LoggerFactory.getLogger(FileController.class);

    @Autowired
    private FileService fileService;
    @Autowired
    private FileConfigProperties configProperties;

    /**
     * 上传单文件
     *
     * @param request HttpServletRequest
     * @param file    文件
     * @return 文件信息
     * @ignoreParams request
     */
    @ApiOperation(value = "上传单文件")
    @ApiOperationSupport(order = 1)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "file", value = "要上传的文件", required = true),
            @ApiImplicitParam(name = "_child", value = "子文件标识", paramType = "form"),
    })
    @PostMapping(value = "/upload")
    public ApiResult<FileObjRespVO<?>> upload(HttpServletRequest request, MultipartFile file) {
        if (Boolean.FALSE.equals(configProperties.getUploadEnabled())) {
            return ApiResult.fail("请使用文件服务进行上传");
        }
        return fileService.upload(file, ServletUtil.getParams(request));
    }

    @ApiOperation(value = "批量上传文件（小文件）")
    @ApiOperationSupport(order = 1)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "file", value = "要上传的文件", required = true),
            @ApiImplicitParam(name = "_child", value = "子文件标识", paramType = "form"),
    })
    @PostMapping(value = "/upload/batch")
    public ApiResult<List<FileObjRespVO<?>>> uploadBatch(HttpServletRequest request, MultipartFile[] files) {
        if (Boolean.FALSE.equals(configProperties.getUploadEnabled())) {
            return ApiResult.fail("请使用文件服务进行上传");
        }

        if (ArrayUtil.isEmpty(files)) {
            return ApiResult.fail("请选择要上传的文件");
        }
        var params = ServletUtil.getParams(request);

        List<FileObjRespVO<?>> resultList = new ArrayList<>();
        ApiResult<FileObjRespVO<?>> result = null;
        String msg = null;
        for (MultipartFile file : files) {
            result = fileService.upload(file, params);
            if (result.getData() != null) {
                resultList.add(result.getData());
            } else {
                msg = result.getMsg();
                log.error("上传文件失败：{}", msg);
            }
        }
        if (resultList.isEmpty()) {
            return ApiResult.fail(msg);
        }
        return ApiResult.ok(resultList);
    }

    /**
     * 请求分片上传
     *
     * @param reqVO 文件信息
     * @return 文件信息
     */
    @ApiOperation(value = "请求分片上传")
    @ApiOperationSupport(order = 3)
    @PostMapping(value = "/apply/chunk")
    public ApiResult<Long> applyChunk(@RequestBody FileChunkReqVO reqVO) {
        if (Boolean.FALSE.equals(configProperties.getUploadEnabled())) {
            return ApiResult.fail("请使用文件服务进行上传");
        }
        return fileService.applyChunk(reqVO);
    }

    /**
     * 分片上传
     *
     * @param saveVO 分片信息
     * @param file   文件
     * @return 文件信息
     */
    @ApiOperation(value = "分片上传")
    @ApiOperationSupport(order = 4)
    @PostMapping(value = "/upload/chunk")
    public ApiResult<FileObjRespVO<?>> uploadChunk(@RequestParam(value = "file", required = false) MultipartFile file,
                                                   FileChunkSaveVO saveVO) {
        if (Boolean.FALSE.equals(configProperties.getUploadEnabled())) {
            return ApiResult.fail("请使用文件服务进行上传");
        }
        return fileService.uploadChunk(file, saveVO);
    }

    /**
     * 查询单个文件信息
     *
     * @param fileCode 文件编码
     * @return 文件信息
     */
    @ApiOperation(value = "查询单个文件信息")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "fileCode", value = "文件唯一标识", required = true)
    })
    @GetMapping(value = "/{fileCode}")
    public ApiResult<FileObjRespVO<?>> get(@PathVariable String fileCode) {
        return fileService.get(fileCode);
    }

    /**
     * 查询单个文件信息
     *
     * @param fileCodes 文件编码
     * @return 文件信息列表
     */
    @ApiOperation(value = "查询多个文件信息")
    @ApiOperationSupport(order = 6)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "fileCodes", value = "文件唯一标识", required = true, dataTypeClass = String.class, allowMultiple = true)
    })
    @PostMapping(value = "/query")
    public ApiResult<List<FileObjRespVO<?>>> query(@RequestBody List<String> fileCodes) {
        return fileService.query(fileCodes);
    }

    /**
     * 显示图片
     *
     * @param fileCode  文件编码
     * @param thumbnail 是否是预览缩略图
     * @param response  HttpServletResponse
     * @return 图片流
     * @download
     * @ignoreParams response
     */
    @ApiOperation(value = "显示图片")
    @ApiOperationSupport(order = 7)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "fileCode", value = "文件唯一标识", required = true),
            @ApiImplicitParam(name = "thumbnail", value = "是否预览缩略图", defaultValue = "true"),
    })
    @GetMapping(value = "/{fileCode}/show")
    public HttpEntity<Resource> show(@PathVariable String fileCode, Boolean thumbnail, HttpServletResponse response) {
        if (Boolean.FALSE.equals(configProperties.getDownloadEnabled())) {
            log.error("请使用文件服务进行下载");
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
        }
        HttpEntity<Resource> httpEntity = null;
        try {
            httpEntity = fileService.preview(fileCode, thumbnail != null && thumbnail ? ConstantsFile.PARAM_FILE_CHILD_IMG_THUMBNAIL : null);
        } catch (Exception e) {
            log.error("预览图片异常：", e);
            return ResponseEntity.badRequest().build();
        }

        if (httpEntity.getHeaders().getContentType() == null) {
            // 默认图片类型
            response.setContentType(MediaType.IMAGE_JPEG_VALUE);
        }

        response.setHeader(HttpHeaders.CACHE_CONTROL, CacheControl.maxAge(Duration.ofDays(7)).cachePublic().getHeaderValue());
        return httpEntity;
    }

    /**
     * 下载文件
     *
     * @param fileCode 文件编码
     * @return 文件流
     * @download
     */
    @ApiOperation(value = "下载文件")
    @ApiOperationSupport(order = 8)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "fileCode", value = "文件唯一标识", required = true)
    })
    @GetMapping(value = "/{fileCode}/download", produces = MediaType.APPLICATION_OCTET_STREAM_VALUE)
    public HttpEntity<StreamingResponseBody> download(@PathVariable String fileCode) {
        if (Boolean.FALSE.equals(configProperties.getDownloadEnabled())) {
            log.error("请使用文件服务进行下载");
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
        }
        HttpEntity<StreamingResponseBody> httpEntity = null;
        try {
            httpEntity = fileService.download(fileCode, null);
        } catch (Exception e) {
            log.error("下载文件异常：{}", fileCode, e);
            return ResponseEntity.badRequest().build();
        }
        return httpEntity;
    }


    @ApiOperation(value = "打包下载")
    @ApiOperationSupport(order = 9)
    @PostMapping(value = "/download/package", produces = "application/zip")
    public ResponseEntity<StreamingResponseBody> downloadByPackage(@RequestBody @Valid FilePackageParam param) {
        return fileService.downloadByPackage(param);
    }

    @ApiOperation(value = "申请打包下载", notes = "适应于文件大的")
    @ApiOperationSupport(order = 10)
    @PostMapping(value = "/package/apply")
    public ApiResult<Long> applyPackage(@RequestBody @Valid FilePackageParam param) {
        return fileService.applyPackage(param);
    }

    @ApiOperation(value = "查询打包结果", notes = "打包结束且成功的方可下载")
    @ApiOperationSupport(order = 11)
    @ApiImplicitParam(name = "applyId", value = "申请记录ID", required = true, paramType = "query")
    @GetMapping(value = "/package/result")
    public ApiResult<PackageResultVO> packageResult(@RequestParam(name = "applyId") @NotNull(message = "申请记录ID为空") Long applyId) {
        return fileService.packageResult(applyId);
    }

    @ApiOperation(value = "下载打包文件")
    @ApiOperationSupport(order = 12)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applyId", value = "申请记录ID", required = true, paramType = "query"),
            @ApiImplicitParam(name = "packageName", value = "下载的压缩包名称", required = false, paramType = "query"),
    })
    @GetMapping(value = "/package/apply/download", produces = "application/zip")
    public ResponseEntity<StreamingResponseBody> downloadPackage(@RequestParam(name = "applyId") @NotNull(message = "申请记录ID为空") Long applyId,
                                                                 @RequestParam(name = "applyId", required = false) String packageName) {
        return fileService.downloadPackage(applyId, packageName);
    }

    /**
     * 删除文件
     *
     * @param fileCode 文件编码
     * @return 文件编码
     */
    @ApiOperation(value = "删除文件")
    @ApiOperationSupport(order = 21)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "fileCode", value = "文件唯一标识", required = true)
    })
    @DeleteMapping(value = "/{fileCode}")
    public ApiResult<String> delete(@PathVariable String fileCode) {
        return fileService.delete(fileCode);
    }

    /**
     * 批量删除文件
     *
     * @param fileCodes 文件编码
     * @return 文件编码
     */
    @ApiOperation(value = "批量删除文件")
    @ApiOperationSupport(order = 22)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "fileCodes", value = "文件唯一标识", required = true, dataTypeClass = String.class, allowMultiple = true)
    })
    @DeleteMapping(value = "/delete")
    public ApiResult<List<String>> deleteByIdBatch(@RequestBody List<String> fileCodes) {
        return fileService.delete(fileCodes);
    }
}
