package com.el.coordinator.boot.fsm.service;

import com.el.coordinator.boot.fsm.model.dto.FileObjDTO;
import com.el.coordinator.boot.fsm.model.vo.FileObjRespVO;
import com.el.coordinator.core.common.api.ApiResult;
import com.el.coordinator.file.parameter.FilePackageParam;
import com.el.coordinator.file.vo.FileChunkReqVO;
import com.el.coordinator.file.vo.FileChunkSaveVO;
import com.el.coordinator.file.vo.PackageResultVO;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpEntity;
import org.springframework.http.ResponseEntity;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import javax.annotation.Nullable;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.io.File;
import java.io.OutputStream;
import java.util.List;
import java.util.Map;

/**
 * 文件服务接口.
 *
 * @author Kaiser（wang shao）
 * @date 2021-04-16
 */
public interface FileService<T> {

    /**
     * 文件上传
     * <p>
     * 可用于接收controller层上传文件
     *
     * @param uploadFile 上传的文件
     * @param reqParams  其它请求参数
     * @return 上传结果
     */
    ApiResult<FileObjRespVO<T>> upload(MultipartFile uploadFile, Map<String, String[]> reqParams);

    /**
     * 上传文件
     * <p>
     * 可用于上传后端其它服务动态生成的文件
     *
     * @param file 上传的文件
     * @return 文件信息
     */
    ApiResult<FileObjRespVO<T>> upload(File file);

    /**
     * 上传文件
     * <p>
     * 可用于上传后端其它服务动态生成的文件
     *
     * @param file     上传的文件
     * @param fileName 文件名称，含后缀
     * @return 文件信息
     */
    ApiResult<FileObjRespVO<T>> upload(byte[] file, String fileName);

    /**
     * 申请分片上传
     *
     * @param reqVO 申请信息
     * @return 分片上传记录标识
     */
    ApiResult<Long> applyChunk(@NotNull FileChunkReqVO reqVO);

    /**
     * 开始分片上传
     *
     * @param file   分片文件，合并时可为空
     * @param saveVO 分片信息
     * @return 文件信息
     */
    ApiResult<FileObjRespVO<T>> uploadChunk(@Nullable MultipartFile file, @NotNull FileChunkSaveVO saveVO);

    /**
     * 下载文件
     *
     * @param fileCode  文件唯一标识
     * @param childFlag 子文件，如果不传则下载原文件
     * @return 下载结果
     */
    HttpEntity<StreamingResponseBody> download(String fileCode, String childFlag);

    /**
     * 打包下载
     *
     * @param param 打包参数
     * @return 文件流
     */
    ResponseEntity<StreamingResponseBody> downloadByPackage(FilePackageParam param);

    /**
     * 申请下载打包文件
     *
     * @param param 打包文件参数
     * @return 申请记录ID
     */
    ApiResult<Long> applyPackage(FilePackageParam param);

    /**
     * 查询打包结果
     *
     * @param applyId 申请记录ID
     * @return 打包结果
     */
    ApiResult<PackageResultVO> packageResult(@NotNull Long applyId);

    /**
     * 下载打包文件
     *
     * @param applyId 申请记录ID
     * @return 文件流
     */
    ResponseEntity<StreamingResponseBody> downloadPackage(@NotNull Long applyId, String packageName);

    /**
     * 预览图片
     *
     * @param fileCode  文件唯一标识
     * @param childFlag 显示子文件，如果不传则显示原
     * @return 下载路径
     */
    HttpEntity<Resource> preview(String fileCode, String childFlag);

    /**
     * 预览图片
     *
     * @param fileCode     文件唯一标识
     * @param outputStream 输出流
     * @param childFlag    显示子文件，如果不传则显示原
     * @return 下载路径
     * @deprecated 不稳定
     */
    @Deprecated(since = "1.2.0", forRemoval = true)
    ApiResult<String> preview(String fileCode, OutputStream outputStream, String childFlag);

    /**
     * 删除文件
     *
     * @param fileCode 文件唯一标识
     * @return 删除结果
     */
    ApiResult<String> delete(String fileCode);

    /**
     * 删除文件
     *
     * @param fileCodes 文件唯一标识
     * @return 删除结果
     */
    ApiResult<List<String>> delete(List<String> fileCodes);

    /**
     * 判断是否存在
     *
     * @param fileCode 文件唯一标识
     * @return 是否存在
     */
    ApiResult<Boolean> existsAll(String fileCode);

    /**
     * 判断是否存在
     *
     * @param fileCodes 文件唯一标识
     * @return 是否存在
     */
    ApiResult<Boolean> existsAll(List<String> fileCodes);

    /**
     * 获取文件信息
     *
     * @param fileCode 文件唯一标识
     * @return 文件信息
     */
    ApiResult<FileObjRespVO<T>> get(String fileCode);

    /**
     * 获取文件信息
     *
     * @param fileCode 文件唯一标识
     * @return 文件信息
     */
    ApiResult<FileObjDTO<T>> getForDto(String fileCode);

    /**
     * 查询文件信息
     *
     * @param fileCodes 文件唯一标识
     * @return 文件信息
     */
    ApiResult<List<FileObjRespVO<T>>> query(List<String> fileCodes);

    /**
     * 查询文件信息
     *
     * @param fileCodes 文件唯一标识
     * @return 文件信息
     */
    ApiResult<List<FileObjDTO<T>>> queryDto(List<String> fileCodes);
}
