package com.el.coordinator.boot.fsm.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.exceptions.ExceptionUtil;
import cn.hutool.core.io.IoUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.el.coordinator.boot.fsm.common.ConstantsFile;
import com.el.coordinator.boot.fsm.common.UploadFileParam;
import com.el.coordinator.boot.fsm.config.FileConfigProperties;
import com.el.coordinator.boot.fsm.config.child.ChildProcessor;
import com.el.coordinator.boot.fsm.config.handler.Handlable;
import com.el.coordinator.boot.fsm.config.validator.Validatable;
import com.el.coordinator.boot.fsm.model.dto.FileObjDTO;
import com.el.coordinator.boot.fsm.model.vo.FileObjRespVO;
import com.el.coordinator.boot.fsm.service.FileService;
import com.el.coordinator.boot.fsm.service.storage.FileStorageService;
import com.el.coordinator.boot.fsm.util.FileUploadUtil;
import com.el.coordinator.core.common.api.ApiResult;
import com.el.coordinator.core.common.exception.BusinessException;
import com.el.coordinator.file.parameter.FilePackageParam;
import com.el.coordinator.file.vo.FileChunkReqVO;
import com.el.coordinator.file.vo.FileChunkSaveVO;
import com.el.coordinator.file.vo.PackageResultVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpEntity;
import org.springframework.http.ResponseEntity;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021-04-18
 */
@Slf4j
public class FileServiceImpl<T> implements FileService<T> {

    private final FileConfigProperties configProperties;
    private final FileStorageService<T> storageService;
    private final List<Validatable<T>> validatableList;
    private final Handlable<T> handlable;
    private final List<ChildProcessor<T>> childProcessorList;

    public FileServiceImpl(FileConfigProperties configProperties, FileStorageService<T> storageService,
                           List<Validatable<T>> validatableList,
                           List<ChildProcessor<T>> childProcessorList, Handlable<T> handlable) {
        this.configProperties = configProperties;
        this.storageService = storageService;
        this.validatableList = validatableList;
        this.childProcessorList = childProcessorList;
        this.handlable = handlable;
    }

    @Override
    public ApiResult<FileObjRespVO<T>> upload(MultipartFile uploadFile, Map<String, String[]> reqParams) {
        UploadFileParam<T> uploadFileParam = null;
        try {
            uploadFileParam = wrapperUploadFile(uploadFile == null ? null : uploadFile.getResource(), reqParams);
        } catch (Exception e) {
            log.error("上传文件失败：", e);
            return ApiResult.fail(ExceptionUtil.getRootCause(e).getMessage());
        }

        return upload(uploadFileParam);
    }

    @Override
    public ApiResult<FileObjRespVO<T>> upload(File file) {
        UploadFileParam<T> uploadFileParam = null;
        try {
            uploadFileParam = wrapperUploadFile(new FileSystemResource(file), Collections.emptyMap());
        } catch (Exception e) {
            log.error("上传文件失败：", e);
            return ApiResult.fail(ExceptionUtil.getRootCause(e).getMessage());
        }

        return upload(uploadFileParam);
    }

    @Override
    public ApiResult<FileObjRespVO<T>> upload(byte[] file, String fileName) {
        if (file == null || file.length == 0) {
            return ApiResult.fail("上传文件为空");
        }

        UploadFileParam<T> uploadFileParam = null;
        try {
            uploadFileParam = wrapperUploadFile(byte2Resource(file, fileName), Collections.emptyMap());
        } catch (Exception e) {
            log.error("上传文件失败：", e);
            return ApiResult.fail(ExceptionUtil.getRootCause(e).getMessage());
        }

        return upload(uploadFileParam);
    }

    @Override
    public ApiResult<Long> applyChunk(FileChunkReqVO reqVO) {
        var size = reqVO.getSize();
        if (size < configProperties.getChunkMin().toBytes()) {
            return ApiResult.fail("文件太小，请勿使用分片上传");
        }
        var uploadId = storageService.applyChunk(reqVO);
        return ApiResult.ok(uploadId);
    }

    @Override
    public ApiResult<FileObjRespVO<T>> uploadChunk(@Nullable MultipartFile file, FileChunkSaveVO saveVO) {
        boolean isMerge = Boolean.TRUE.equals(saveVO.getMerge());
        FileObjDTO<T> fileObj = null;
        if (isMerge) {
            fileObj = storageService.uploadChunk(null, saveVO);
            if (fileObj == null) {
                return ApiResult.fail("上传失败");
            }
            var fileRespVO = handlable.uploadDtoToRespVo(fileObj);
            return ApiResult.ok(fileRespVO);
        }

        // 上传分片时
        if (file == null || file.isEmpty()) {
            return ApiResult.fail("上传的文件为空");
        }
        if (saveVO.getPartNumber() == null) {
            return ApiResult.fail("分片序号为空");
        }
        if (saveVO.getPartNumber() < 0 || saveVO.getPartNumber() > 10000) {
            return ApiResult.fail("分片序号有效范围：0 ~ 10000");
        }
        var size = file.getSize();
        if (size > configProperties.getChunkMax().toBytes()) {
            return ApiResult.fail("分片大小取值范围：" + configProperties.getChunkMin().toMegabytes() + "MB ~ " + configProperties.getChunkMax().toMegabytes() + "MB");
        }
        storageService.uploadChunk(file.getResource(), saveVO);
        // 上传时无需返回文件信息
        return ApiResult.ok();
    }

    @Override
    public HttpEntity<StreamingResponseBody> download(String fileCode, String childFlag) {
        if (StrUtil.isBlank(fileCode)) {
            return ResponseEntity.badRequest().build();
        }

        try {
            return storageService.downloadStreaming(fileCode, ObjectUtil.defaultIfNull(childFlag, "DEFAULT"));
        } catch (Exception e) {
            log.error("下载文件失败：", e);
        }

        return ResponseEntity.badRequest().build();
    }

    @Override
    public ResponseEntity<StreamingResponseBody> downloadByPackage(FilePackageParam param) {
        if (param == null || CollUtil.isEmpty(param.getFileList())) {
            return ResponseEntity.badRequest().build();
        }

        try {
            return storageService.downloadByPackage(param);
        } catch (Exception e) {
            log.error("下载文件失败", e);
        }
        return ResponseEntity.badRequest().build();
    }

    @Override
    public ApiResult<Long> applyPackage(FilePackageParam param) {
        if (param == null || CollUtil.isEmpty(param.getFileList())) {
            return ApiResult.fail("下载文件标识为空");
        }

        return storageService.applyPackage(param);
    }

    @Override
    public ApiResult<PackageResultVO> packageResult(Long applyId) {
        if (applyId == null) {
            return ApiResult.fail("申请记录ID为空");
        }

        return storageService.packageResult(applyId);
    }

    @Override
    public ResponseEntity<StreamingResponseBody> downloadPackage(Long applyId, String packageName) {
        if (applyId == null) {
            log.error("申请记录ID为空");
            return ResponseEntity.noContent().build();
        }

        return storageService.downloadPackage(applyId, packageName);
    }

    @Override
    public HttpEntity<Resource> preview(String fileCode, String childFlag) {
        if (StrUtil.isBlank(fileCode)) {
            return ResponseEntity.badRequest().build();
        }

        try {
            return storageService.download(fileCode, ObjectUtil.defaultIfNull(childFlag, "DEFAULT"));
        } catch (Exception e) {
            log.error("下载文件失败：", e);
        }

        return ResponseEntity.badRequest().build();
    }

    @Override
    public ApiResult<String> preview(String fileCode, OutputStream outputStream, String childFlag) {
        if (StrUtil.isBlank(fileCode)) {
            return ApiResult.fail("文件的唯一标识为空");
        }

        try {
            var resource = storageService.download(fileCode, ObjectUtil.defaultIfNull(childFlag, "DEFAULT")).getBody();
            if (resource != null && resource.isReadable()) {
                IoUtil.copy(resource.getInputStream(), outputStream);
                return null;
            }
        } catch (Exception e) {
            log.error("下载文件失败：", e);
        }

        return ApiResult.fail("预览文件失败");
    }

    @Override
    public ApiResult<String> delete(String fileCode) {
        if (StrUtil.isBlank(fileCode)) {
            return ApiResult.fail("删除失败，文件的唯一标识为空");
        }

        try {
            storageService.delete(fileCode);
        } catch (Exception e) {
            log.error("删除文件失败：", e);
            return ApiResult.fail(ExceptionUtil.getRootCause(e).getMessage());
        }
        return ApiResult.ok(fileCode);
    }

    @Override
    public ApiResult<List<String>> delete(List<String> fileCodes) {
        if (CollectionUtil.isEmpty(fileCodes)) {
            return ApiResult.fail("删除失败，文件的唯一标识为空");
        }

        try {
            storageService.delete(fileCodes);
        } catch (Exception e) {
            log.error("删除文件失败：", e);
            return ApiResult.fail(ExceptionUtil.getRootCause(e).getMessage());
        }
        return ApiResult.ok(fileCodes);
    }

    @Override
    public ApiResult<Boolean> existsAll(String fileCode) {
        if (StrUtil.isBlank(fileCode)) {
            return ApiResult.fail("文件的唯一标识为空");
        }

        try {
            var res = storageService.exists(fileCode);
            return ApiResult.ok(res);
        } catch (Exception e) {
            log.error("判断文件是否存在失败：", e);
            return ApiResult.fail(ExceptionUtil.getRootCause(e).getMessage());
        }
    }

    @Override
    public ApiResult<Boolean> existsAll(List<String> fileCodes) {
        if (CollectionUtil.isEmpty(fileCodes)) {
            return ApiResult.fail("文件的唯一标识为空");
        }

        try {
            var res = storageService.exists(fileCodes);
            return ApiResult.ok(res);
        } catch (Exception e) {
            log.error("判断文件是否存在失败：", e);
            return ApiResult.fail("判断文件是否存在失败：" + e.getMessage());
        }
    }

    @Override
    public ApiResult<FileObjRespVO<T>> get(String fileCode) {
        if (StrUtil.isBlank(fileCode)) {
            return ApiResult.fail("文件的唯一标识为空");
        }

        FileObjDTO<T> fileObjDTO = null;
        try {
            fileObjDTO = storageService.getByFileCode(fileCode);
        } catch (Exception e) {
            log.error("获取文件是否存在失败：", e);
            return ApiResult.fail(ExceptionUtil.getRootCause(e).getMessage());
        }
        if (fileObjDTO == null) {
            return ApiResult.fail("文件不存在");
        }

        return ApiResult.ok(handlable.uploadDtoToRespVo(fileObjDTO));
    }

    @Override
    public ApiResult<FileObjDTO<T>> getForDto(String fileCode) {
        if (StrUtil.isBlank(fileCode)) {
            return ApiResult.fail("文件的唯一标识为空");
        }

        FileObjDTO<T> fileObjDTO = null;
        try {
            fileObjDTO = storageService.getByFileCode(fileCode);
        } catch (Exception e) {
            log.error("获取文件是否存在失败：", e);
            return ApiResult.fail(ExceptionUtil.getRootCause(e).getMessage());
        }
        if (fileObjDTO == null) {
            return ApiResult.fail("文件不存在");
        }

        handlable.fillFileObj(fileObjDTO);
        return ApiResult.ok(fileObjDTO);
    }

    @Override
    public ApiResult<List<FileObjRespVO<T>>> query(List<String> fileCodes) {
        if (CollectionUtil.isEmpty(fileCodes)) {
            return ApiResult.fail("文件的唯一标识为空");
        }

        List<FileObjDTO<T>> files = null;
        try {
            files = storageService.queryByFileCode(fileCodes);
        } catch (Exception e) {
            log.error("判断文件是否存在失败：", e);
            return ApiResult.fail(ExceptionUtil.getRootCause(e).getMessage());
        }
        if (CollectionUtil.isEmpty(files)) {
            return ApiResult.ok(Collections.emptyList());
        }

        List<FileObjRespVO<T>> respVOList = files.stream()
                .map(handlable::uploadDtoToRespVo)
                .collect(Collectors.toList());
        return ApiResult.ok(respVOList);
    }

    @Override
    public ApiResult<List<FileObjDTO<T>>> queryDto(List<String> fileCodes) {
        if (CollectionUtil.isEmpty(fileCodes)) {
            return ApiResult.fail("文件的唯一标识为空");
        }

        List<FileObjDTO<T>> files = null;
        try {
            files = storageService.queryByFileCode(fileCodes);
        } catch (Exception e) {
            log.error("判断文件是否存在失败：", e);
            return ApiResult.fail(ExceptionUtil.getRootCause(e).getMessage());
        }
        if (CollectionUtil.isEmpty(files)) {
            return ApiResult.ok(Collections.emptyList());
        }

        files.parallelStream().forEach(handlable::fillFileObj);
        return ApiResult.ok(files);
    }

    private Resource byte2Resource(byte[] file, String filename) {
        return new ByteArrayResource(file) {

            @Override
            @Nonnull
            public String getDescription() {
                return "byte文件流";
            }

            @Override
            public String getFilename() {
                return filename;
            }
        };
    }

    private ApiResult<FileObjRespVO<T>> upload(UploadFileParam<T> uploadFileParam) {
        // 先校验上传文件
        String msg = validateUploadFile(uploadFileParam);
        if (msg != null) {
            return ApiResult.fail("上传文件失败，" + msg);
        }

        // 文件前置处理器处理
        ApiResult<FileObjRespVO<T>> result = handlable.beforeHandle(uploadFileParam);
        if (result != null) {
            return result;
        }

        // 开始上传文件
        FileObjDTO<T> fileObj = null;
        try {
            fileObj = storageService.upload(handlable.paramToUploadDTO(uploadFileParam));
        } catch (Exception e) {
            log.error("文件存储失败：", e);
            msg = ExceptionUtil.getRootCause(e).getMessage();
            return ApiResult.fail(msg);
        }

        // 处理子文件
        dealChildFile(uploadFileParam, fileObj);

        // 上传后的处理返回结果
        return handlable.afterHandle(uploadFileParam, fileObj, msg);
    }

    /**
     * 包装上传文件的参数
     *
     * @param uploadFile  上传的文件
     * @param uploadParam 上传时的其它参数
     * @return 文件参数
     */
    private UploadFileParam<T> wrapperUploadFile(Resource uploadFile, Map<String, String[]> uploadParam) throws IOException {
        if (uploadFile == null) {
            throw new BusinessException("上传的文件为空");
        }

        return (UploadFileParam<T>) FileUploadUtil.resource2UploadFileParam(uploadFile, uploadParam);
    }

    /**
     * 校验上传的文件
     *
     * @param uploadFileParam 上传的文件信息
     * @return 校验不通过的提示信息
     */
    private String validateUploadFile(UploadFileParam<T> uploadFileParam) {
        if (validatableList.isEmpty()) {
            return null;
        }

        String msg = null;
        for (Validatable<T> validatable : validatableList) {
            msg = validatable.validate(uploadFileParam);
            if (msg != null) {
                return msg;
            }
        }
        return null;
    }

    /**
     * 校验上传的文件
     *
     * @param uploadFileParam 上传的文件信息
     * @param parentFile      上传的父文件信息
     */
    private void dealChildFile(UploadFileParam<T> uploadFileParam, FileObjDTO<T> parentFile) {
        if (childProcessorList.isEmpty()) {
            return;
        }

        String[] childrenFile = uploadFileParam.getUploadParam().get(ConstantsFile.PARAM_NAME_CHILD);
        if (ArrayUtil.isEmpty(childrenFile)) {
            return;
        }

        for (ChildProcessor<T> processor : childProcessorList) {
            for (String child : childrenFile) {
                if (processor.accept(child)) {
                    processor.process(uploadFileParam, parentFile, tFileUploadDTO -> {
                        try {
                            return storageService.upload(tFileUploadDTO);
                        } catch (Exception e) {
                            log.error("处理子文件失败：", e);
                            throw new BusinessException("上传文件失败");
                        }
                    });
                }
            }
        }
    }
}
