package com.el.coordinator.boot.fsm.service.impl.exportstrategy;

import cn.hutool.core.io.FileUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.excel.write.builder.ExcelWriterBuilder;
import com.el.coordinator.boot.fsm.model.bo.ExportStrategyParam;
import com.el.coordinator.boot.fsm.model.vo.FileObjRespVO;
import com.el.coordinator.boot.fsm.service.FileService;
import com.el.coordinator.boot.fsm.service.exportdata.DataExport;
import com.el.coordinator.boot.fsm.support.FsmTmplSupport;
import com.el.coordinator.boot.fsm.util.FileUploadUtil;
import com.el.coordinator.core.common.api.ApiResult;
import com.el.coordinator.core.common.constant.ConstantFsm;
import com.el.coordinator.core.common.exception.BusinessException;
import com.el.coordinator.file.business.dto.ImportRateDTO;
import com.el.coordinator.file.business.dto.TmplDTO;
import com.el.coordinator.file.business.param.ImportResultDTO;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2022/11/28
 */
abstract class BaseExportStrategy implements ExportStrategy {
    private static final Logger LOG = LoggerFactory.getLogger(BaseExportStrategy.class);
    protected static final ObjectMapper OBJECT_MAPPER = FileUploadUtil.createObjectMapper();

    protected final FileService fileService;
    protected final FsmTmplSupport fsmTmplSupport;

    public BaseExportStrategy(FileService fileService, FsmTmplSupport fsmTmplSupport) {
        this.fileService = fileService;
        this.fsmTmplSupport = fsmTmplSupport;
    }

    /**
     * 表头
     *
     * @param tmplDTO
     * @return
     */
    protected List<List<String>> obtainHeaders(TmplDTO tmplDTO) {
        Assert.notEmpty(tmplDTO.getAttributes(), "未获取到模板的头部信息");
        return tmplDTO.getAttributes().subList(0, tmplDTO.getFieldTypeRow() - 1);
    }

    /**
     * 表头
     *
     * @param tmplDTO
     * @return
     */
    protected List<String> obtainAttributes(TmplDTO tmplDTO) {
        Assert.notEmpty(tmplDTO.getAttributes(), "未获取到模板的头部信息");
        return tmplDTO.getAttributes().get(tmplDTO.getFieldTypeRow() - 1);
    }

    /**
     * 更新导出进度
     *
     * @param importId
     * @param rateDTO
     */
    protected void updateRate(long importId, ImportRateDTO rateDTO) {
        fsmTmplSupport.storeImportRate(importId, rateDTO);
    }

    /**
     * 更新导出完毕状态
     *
     * @param param
     * @param numSuc
     * @param msg
     */
    protected void updateExportFinish(ExportStrategyParam param, Long numSuc, String msg) {
        this.updateExportFinish(param, numSuc, msg, null);
    }

    protected void updateExportFinish(ExportStrategyParam param, Long numSuc, String msg, File file) {
        // 更新导入结果
        if (param.getImportId() != null) {
            String fileCode = uploadImportFile(file);
            var importResultDTO = ImportResultDTO.builder()
                    .success(StrUtil.isBlank(msg))
                    .numSuc(numSuc)
                    .failMsg(msg)
                    .fileCode(fileCode)
                    .build();
            fsmTmplSupport.updateImportResult(param.getImportId(), importResultDTO);
        } else {
            LOG.error("更新导入导出记录结果失败：{}", msg);
        }

        // 更新限流信息
        updateLimiter(param.getTmplDTO(), false);
    }

    /**
     * 上传导出的文件
     *
     * @param param
     * @param file
     * @param fileIndex
     */
    protected void updateExportFile(ExportStrategyParam param, File file, int fileIndex) {
        String fileCode = this.uploadImportFile(file);
        fsmTmplSupport.saveExportFile(param.getImportId(), fileCode, fileIndex);
    }

    /**
     * 将业务返回的数据转为字符串列表
     *
     * @param dataList
     * @param fields
     * @return
     */
    protected List<List<String>> convertExportData(List<Serializable> dataList, List<String> fields) {
        return OBJECT_MAPPER.convertValue(dataList, new TypeReference<List<Map<String, Object>>>() {
                }).stream()
                .map(data -> fields.stream().map(field -> obtainValue(data, field)).collect(Collectors.toList()))
                .collect(Collectors.toList());
    }

    private String uploadImportFile(File file) {
        if (file != null) {
            ApiResult<FileObjRespVO> uploadResult = fileService.upload(file);
            // 删除临时文件
            file.delete();
            file.getParentFile().delete();
            if (uploadResult.isSuccess()) {
                return uploadResult.getData().getFileCode();
            } else {
                LOG.error("上传导入导出结果文件失败：{}", uploadResult);
            }
        }
        return null;
    }

    private String obtainValue(Map<String, Object> data, String field) {
        if (StrUtil.isBlank(field)) {
            return "";
        }
        Object value = data.get(field);
        if (value == null) {
            return "";
        }
        if (value instanceof LocalDateTime) {
            return FileUploadUtil.FORMATTER_DATETIME.format((LocalDateTime) value);
        } else if (value instanceof LocalDate) {
            return FileUploadUtil.FORMATTER_DATE.format((LocalDate) value);
        } else if (value instanceof Date) {
            return FileUploadUtil.SDF_DATE.format((Date) value);
        } else if (value instanceof Double) {
            return NumberUtil.decimalFormatMoney((Double) value);
        } else if (value instanceof Float) {
            return NumberUtil.decimalFormatMoney((Float) value);
        }
        return value.toString();
    }

    /**
     * 创建导出临时文件
     *
     * @return
     */
    protected File createExportFile(ExportStrategyParam param) {
       return createExportFile(param, null);
    }

    /**
     * 创建导出临时文件
     *
     * @param param
     * @param order
     * @return
     */
    protected File createExportFile(ExportStrategyParam param, Integer order) {
        String fileName = CharSequenceUtil.blankToDefault(param.getDataExport().exportFileName(), param.getTmplDTO().getName());
        if (order != null) {
            fileName = fileName + "_" + order;
        }

        File file = new File(System.getProperty("java.io.tmpdir"), fileName + ".xlsx");
        file.delete();
        try {
            file.createNewFile();
        } catch (IOException e) {
            throw new RuntimeException("创建临时文件失败", e);
        }
        return file;
    }

    /**
     * 获取sheet的数据量限制
     *
     * @param tmplDTO
     * @return
     */
    protected int sheetLimit(TmplDTO tmplDTO) {
        Integer limit = tmplDTO.getExportSheetLimit();
        if (limit == null) {
            return ConstantFsm.SHEET_ROW_MAX;
        }
        if (limit > 0 && limit <= ConstantFsm.SHEET_ROW_MAX) {
            return limit;
        }

        throw new IllegalArgumentException(tmplDTO.getCode() + "参数" + limit + "不合法");
    }

    /**
     * 分页大小
     *
     * @param dataExportService
     * @return
     */
    protected Integer obtainPageSize(DataExport<Serializable, Serializable> dataExportService) {
        Integer pageSize = dataExportService.pageSize();
        if (pageSize == null || pageSize < 1) {
            // 默认页大小
            return 500;
        }

        return pageSize;
    }

    private void loadTmpl(String tmplCode, ExcelWriterBuilder writeBuilder) {
        var tmplResp = fsmTmplSupport.downloadByCode(tmplCode);
        var tmplResource = tmplResp.getBody();
        if (tmplResource == null) {
            throw new BusinessException("未查询到模板文件");
        }
        try {
            writeBuilder.withTemplate(tmplResource.getInputStream());
        } catch (IOException e) {
            LOG.warn("加载导出模板失败");
            return;
        }
    }

    private boolean updateLimiter(TmplDTO tmplDTO, boolean add) {
        if (tmplDTO.getConcurrentLimit() == null || tmplDTO.getConcurrentLimit() == -1) {
            // 不限流
            return true;
        }
        return FsmTmplSupport.updateLimiter(tmplDTO.getCode(), tmplDTO.getConcurrentLimit().toString(), 1, add);
    }
}
