package com.el.coordinator.boot.fsm.service.storage.local;

import cn.hutool.core.io.FileUtil;
import cn.hutool.core.io.IoUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.lang.Snowflake;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.el.coordinator.boot.fsm.config.FileConfigProperties;
import com.el.coordinator.boot.fsm.model.dto.FileObjDTO;
import com.el.coordinator.boot.fsm.model.dto.FileUploadDTO;
import com.el.coordinator.boot.fsm.service.storage.FileStorageService;
import com.el.coordinator.boot.fsm.util.FileUploadUtil;
import com.el.coordinator.core.common.api.ApiResult;
import com.el.coordinator.file.parameter.FilePackageParam;
import com.el.coordinator.file.vo.FileChunkReqVO;
import com.el.coordinator.file.vo.FileChunkSaveVO;
import com.el.coordinator.file.vo.PackageResultVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.springframework.core.io.InputStreamResource;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpEntity;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StreamUtils;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import javax.annotation.Nullable;
import java.io.*;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 文件本地存储服务.
 *
 * @author Kaiser（wang shao）
 * @date 2021-04-18
 */
@Slf4j
public class LocalFileStorageServiceImpl<T> implements FileStorageService<T> {

    private final FileConfigProperties.StorageLocal storageLocal;
    private final Snowflake snowflake = new Snowflake(1, 1);

    private File storageDir = null;

    public LocalFileStorageServiceImpl(FileConfigProperties.StorageLocal storageLocal) {
        this.storageLocal = storageLocal;
        init();
    }

    @Override
    public FileObjDTO<T> upload(FileUploadDTO<T> uploadDTO) throws Exception {
        checkForUpload(uploadDTO);
        String fileCode = snowflake.nextIdStr();

        var resource = uploadDTO.getFileParam().getUploadFile();
        String filename = fileCode + ObjectUtil.defaultIfBlank("." + FileUtil.extName(resource.getFilename()), "");

        // 将resource写入本地文件
        write2File(uploadDTO, filename, fileCode);

        return convert2FileObjDTO(uploadDTO, fileCode);
    }

    @Override
    public Long applyChunk(FileChunkReqVO reqVO) {
        throw new IllegalStateException("暂不支持");
    }

    @Override
    public FileObjDTO<T> uploadChunk(@Nullable Resource file, FileChunkSaveVO saveVO) {
        throw new IllegalStateException("暂不支持");
    }

    @Override
    public HttpEntity<Resource> download(String fileCode, String childFlag) throws Exception {
        var file = findFile(fileCode, childFlag);
        if (file == null) {
            return ResponseEntity.notFound().build();
        }

        try (var inputStream = new FileInputStream(file)){
            return ResponseEntity.ok(new InputStreamResource(inputStream));
        }
    }

    @Override
    public HttpEntity<StreamingResponseBody> downloadStreaming(String fileCode, String childFlag) throws Exception {
        var file = findFile(fileCode, childFlag);
        if (file == null) {
            return ResponseEntity.notFound().build();
        }

        try (var inputStream = new FileInputStream(file)){
            StreamingResponseBody responseBody = outputStream -> {
                StreamUtils.copy(inputStream, outputStream);
                outputStream.close();
            };
            return ResponseEntity.ok(responseBody);
        }
    }

    @Override
    public ResponseEntity<StreamingResponseBody> downloadByPackage(FilePackageParam param) {
        throw new IllegalStateException("暂不支持");
    }

    @Override
    public ApiResult<Long> applyPackage(FilePackageParam param) {
        throw new IllegalStateException("暂不支持");
    }

    @Override
    public ApiResult<PackageResultVO> packageResult(Long applyId) {
        throw new IllegalStateException("暂不支持");
    }

    @Override
    public ResponseEntity<StreamingResponseBody> downloadPackage(Long applyId, String packageName) {
        throw new IllegalStateException("暂不支持");
    }

    @Override
    public void delete(String fileCode) throws Exception {
        var file = findFile(fileCode, null);
        FileUtil.del(file);
    }

    @Override
    public void delete(List<String> fileCodes) throws Exception {
        for (String fileCode : fileCodes) {
            delete(fileCode);
        }
    }

    @Override
    public boolean exists(String fileCode) throws Exception {
        var file = findFile(fileCode, null);

        return file != null;
    }

    @Override
    public boolean exists(List<String> fileCodes) throws Exception {
        for (var fileCode : fileCodes) {
            if (!exists(fileCode)) {
                return false;
            }
        }
        return true;
    }

    @Override
    public FileObjDTO<T> getByFileCode(String fileCode) throws Exception {
        var file = findFile(fileCode, null);
        if (file == null) {
            return null;
        }
        return convert2FileObjDTO(file);
    }

    @Override
    public List<FileObjDTO<T>> queryByFileCode(List<String> fileCodes) throws Exception {

        File[] files = storageDir.listFiles((dir, name) -> {
            for (String fileCode:fileCodes) {
                if (name.startsWith(fileCode)) {
                    return true;
                }
            }
            return false;
        });
        if (ArrayUtil.isNotEmpty(files)) {
            return Arrays.stream(files).flatMap(t -> {
                if (t.isFile()) {
                    return List.of(t).stream();
                }

                File[] children = t.listFiles();
                if (children == null || children.length == 0) {
                    return Stream.empty();
                }
                return Arrays.stream(children);
            }).map(this::convert2FileObjDTO)
                    .collect(Collectors.toList());
        }

        return Collections.emptyList();
    }

    private void init() {
        String path = storageLocal.getPath();
        Assert.notBlank(path, "本地文件服务创建失败，path为空");

        storageDir = new File(path);
        if (!storageDir.exists() && !storageDir.mkdirs()) {
            throw new RuntimeException("创建文件夹失败：" + path);
        }
    }

    private void checkForUpload(FileUploadDTO<T> uploadDTO) {
        if (StrUtil.isNotBlank(uploadDTO.getParentFileCode())) {
            Assert.notBlank(uploadDTO.getChildFlag(), "子文件标识为空");
        }
    }

    /**
     * 将资源写入本地文件
     *
     * @param uploadDTO 资源信息
     * @param fileName  本地文件名称
     * @param fileCode  本地文件唯一标识
     */
    private void write2File(FileUploadDTO<T> uploadDTO, String fileName, String fileCode) {
        File newFile = null;
        if (StrUtil.isBlank(uploadDTO.getParentFileCode())) {
            newFile = new File(storageDir, fileName);
        } else {
            File dir = new File(storageDir, uploadDTO.getParentFileCode() + "_" + uploadDTO.getChildFlag());
            if (!dir.exists()) {
                dir.mkdirs();
            }
            newFile = new File(dir, fileName);
        }

        try {
            IoUtil.copy(uploadDTO.getFileParam().getUploadFile().getInputStream(), new FileOutputStream(newFile));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private FileObjDTO<T> convert2FileObjDTO(FileUploadDTO<T> uploadDTO, String fileCode) {
        FileObjDTO<T> fileObjDTO = new FileObjDTO<>();
        fileObjDTO.setFileCode(fileCode);
        fileObjDTO.setOriginalName(uploadDTO.getFileParam().getOriginalName());
        fileObjDTO.setFileSize(uploadDTO.getFileParam().getFileSize().toBytes());
        fileObjDTO.setSuffix(uploadDTO.getFileParam().getSuffix());
        fileObjDTO.setMimeType(uploadDTO.getFileParam().getMimeType());
        fileObjDTO.setFileType(uploadDTO.getFileParam().getFileType());
        fileObjDTO.setUploadTime(LocalDateTime.now());
        fileObjDTO.setAttribute1(uploadDTO.getAttribute1());
        fileObjDTO.setAttribute2(uploadDTO.getAttribute2());
        fileObjDTO.setAttribute3(uploadDTO.getAttribute3());
        fileObjDTO.setAttribute4(uploadDTO.getAttribute4());
        fileObjDTO.setAttribute5(uploadDTO.getAttribute5());
        return fileObjDTO;
    }

    private FileObjDTO<T> convert2FileObjDTO(File file) {
        FileObjDTO<T> fileObjDTO = new FileObjDTO<>();
        String filename = file.getName();

        int dotIndex = filename.lastIndexOf(".");
        if (dotIndex < 0) {
            fileObjDTO.setFileCode(filename);
        } else {
            fileObjDTO.setFileCode(filename.substring(0, dotIndex));
        }
        fileObjDTO.setOriginalName(filename);
        fileObjDTO.setFileSize(file.length());
        fileObjDTO.setSuffix(FileUtil.extName(filename));
        fileObjDTO.setMimeType(FileUtil.getMimeType(filename));
        fileObjDTO.setFileType(FileUploadUtil.obtainFileType(fileObjDTO.getMimeType()));
        fileObjDTO.setUploadTime(LocalDateTime.ofInstant(Instant.ofEpochMilli(file.lastModified()), ZoneId.systemDefault()));

        return fileObjDTO;
    }

    private File findFile(String fileCode, String childFlag) {
        if (StrUtil.isBlank(childFlag)) {
            File[] files = storageDir.listFiles((dir, name) -> name.startsWith(fileCode));
            if (files == null || files.length == 0) {
                return null;
            }
            return files[0];
        }

        File dir = new File(storageDir, fileCode + "_" + childFlag);
        File[] files = dir.listFiles();
        if (files == null || files.length == 0) {
            return null;
        }
        return files[0];
    }
}
