package com.el.coordinator.boot.fsm.service.storage.remote;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.BooleanUtil;
import cn.hutool.core.util.StrUtil;
import com.el.coordinator.boot.fsm.config.FileConfigProperties;
import com.el.coordinator.boot.fsm.model.dto.FileObjDTO;
import com.el.coordinator.boot.fsm.model.dto.FileUploadDTO;
import com.el.coordinator.boot.fsm.service.storage.FileStorageService;
import com.el.coordinator.boot.fsm.util.FileUploadUtil;
import com.el.coordinator.core.common.api.ApiResult;
import com.el.coordinator.core.common.exception.BusinessException;
import com.el.coordinator.core.common.jpa.vo.PagingVO;
import com.el.coordinator.file.api.FsmApiRequest;
import com.el.coordinator.file.enums.FsmUploadAttributeEnum;
import com.el.coordinator.file.enums.FsmUploadTypeEnum;
import com.el.coordinator.file.parameter.CreatFileUploadInfoParam;
import com.el.coordinator.file.parameter.FilePackageParam;
import com.el.coordinator.file.parameter.FileUploadInfoQueryParam;
import com.el.coordinator.file.parameter.StartFileUploadParam;
import com.el.coordinator.file.vo.*;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpEntity;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import javax.annotation.Nullable;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 远程文件存储服务.
 *
 * @author Kaiser（wang shao）
 * @date 2021-04-18
 */
@Slf4j
public class RemoteFileStorageServiceImpl<T> implements FileStorageService<T> {

    private final FsmApiRequest apiRequest;
    private final FileConfigProperties.StorageRemote configProperties;

    public RemoteFileStorageServiceImpl(FileConfigProperties.StorageRemote configProperties, RestTemplate restTemplate,
                                        ObjectMapper objectMapper) {
        this.configProperties = configProperties;
        this.apiRequest = new FsmApiRequest(configProperties.getServerUrl(), restTemplate, objectMapper);
    }

    @Override
    public FileObjDTO<T> upload(FileUploadDTO<T> uploadDTO) {
        // 第一步，获取授权信息
        var creatFileUploadInfoVO = applyAuthForUpload(uploadDTO);

        // 第二步，开始真正的上传
        var startFileUploadVO = startUpload(uploadDTO.getFileParam().getUploadFile(), creatFileUploadInfoVO);

        return vo2DTO(startFileUploadVO);
    }

    @Override
    public Long applyChunk(FileChunkReqVO reqVO) {
        CreatFileUploadInfoParam uploadInfoParam = new CreatFileUploadInfoParam();
        uploadInfoParam.setTenant("EL");
        uploadInfoParam.setProject(configProperties.getProjectFlag());
        uploadInfoParam.setBusiness("common");
        uploadInfoParam.setAttribute(FsmUploadAttributeEnum.ACTIVITY);
        uploadInfoParam.setType(FsmUploadTypeEnum.FILE);
        uploadInfoParam.setUploaderCode("backend");

        uploadInfoParam.setOriginalFilename(reqVO.getFilename());
        uploadInfoParam.setSize(reqVO.getSize());
        uploadInfoParam.setMimeType(reqVO.getMimeType());

        var result = apiRequest.applyChunk(uploadInfoParam);
        if (result.getData() == null) {
            throw new IllegalArgumentException(CharSequenceUtil.blankToDefault(result.getMsg(), "分片上传失败"));
        }
        return result.getData();
    }

    @Override
    public FileObjDTO<T> uploadChunk(@Nullable Resource file, FileChunkSaveVO saveVO) {
        var result = apiRequest.uploadChunk(file, saveVO);
        if (result == null || result.getData() == null) {
            return null;
        }
        return this.vo2DTO(result.getData());
    }

    @Override
    public HttpEntity<Resource> download(String fileCode, String childFlag) throws Exception {
        return apiRequest.download(fileCode, childFlag);
    }

    @Override
    public HttpEntity<StreamingResponseBody> downloadStreaming(String fileCode, String childFlag) throws Exception {
        return apiRequest.downloadStreaming(fileCode, childFlag);
    }

    @Override
    public ResponseEntity<StreamingResponseBody> downloadByPackage(FilePackageParam param) throws Exception {
        return apiRequest.downloadByPackage(param);
    }

    @Override
    public ApiResult<Long> applyPackage(FilePackageParam param) {
        return apiRequest.applyPackage(param);
    }

    @Override
    public ApiResult<PackageResultVO> packageResult(Long applyId) {
        return apiRequest.packageResult(applyId);
    }

    @Override
    public ResponseEntity<StreamingResponseBody> downloadPackage(Long applyId, String packageName) {
        return apiRequest.downloadPackage(applyId, packageName);
    }

    @Override
    public void delete(String fileCode) throws Exception {
        apiRequest.deleteFile(fileCode);
    }

    @Override
    public void delete(List<String> fileCodes) throws Exception {
        for (String fileCode : fileCodes) {
            apiRequest.deleteFile(fileCode);
        }
    }

    @Override
    public boolean exists(String fileCode) throws Exception {
        var res = apiRequest.exsits(fileCode);
        return res != null && BooleanUtil.isTrue(res.getData());
    }

    @Override
    public boolean exists(List<String> fileCodes) throws Exception {
        var res = apiRequest.exsits(fileCodes);
        return res != null && BooleanUtil.isTrue(res.getData());
    }

    @Override
    public FileObjDTO<T> getByFileCode(String fileCode) throws Exception {
        ApiResult<FileUploadInfoVO> queryResult = apiRequest.findFileCodeOne(fileCode);
        if (!queryResult.isSuccess()) {
            throw new BusinessException(queryResult.getMsg());
        }

        return vo2DTO(queryResult.getData());
    }

    @Override
    public List<FileObjDTO<T>> queryByFileCode(List<String> fileCodes) throws Exception {
        var queryParam = new FileUploadInfoQueryParam<>();
        queryParam.setFileCode(fileCodes);
        queryParam.setSize(fileCodes.size());
        ApiResult<PagingVO<FileUploadInfoVO>> queryResult = apiRequest.search(queryParam);
        if (!queryResult.isSuccess()) {
            throw new BusinessException(queryResult.getMsg());
        }

        if (CollUtil.isEmpty(queryResult.getData().getRecords())) {
            return Collections.emptyList();
        }

        return queryResult.getData().getRecords().parallelStream().map(this::vo2DTO).collect(Collectors.toList());
    }

    /**
     * 上传的第一步，申请授权信息
     *
     * @param uploadDTO 上传参数
     * @return 结果
     */
    private CreatFileUploadInfoVO applyAuthForUpload(FileUploadDTO<T> uploadDTO) {
        var uploadInfoParam = new CreatFileUploadInfoParam<>();
        uploadInfoParam.setTenant("EL");
        uploadInfoParam.setProject(configProperties.getProjectFlag());
        uploadInfoParam.setBusiness("common");
        uploadInfoParam.setAttribute(FsmUploadAttributeEnum.ACTIVITY);
        uploadInfoParam.setType(FsmUploadTypeEnum.FILE);
        uploadInfoParam.setUploaderCode("backend");
        uploadInfoParam.setParentFileCode(StrUtil.blankToDefault(uploadDTO.getParentFileCode(), null));
        uploadInfoParam.setChildFlag(StrUtil.blankToDefault(uploadDTO.getChildFlag(), null));
        uploadInfoParam.setAttribute1(uploadDTO.getAttribute1());
        uploadInfoParam.setAttribute2(uploadDTO.getAttribute2());
        uploadInfoParam.setAttribute3(uploadDTO.getAttribute3());
        uploadInfoParam.setAttribute4(uploadDTO.getAttribute4());
        uploadInfoParam.setAttribute5(uploadDTO.getAttribute5());

        ApiResult<CreatFileUploadInfoVO> authResult = apiRequest.applyFileUploadInfo(uploadInfoParam);
        if (!authResult.isSuccess()) {
            throw new BusinessException(authResult.getMsg());
        }

        return authResult.getData();
    }

    private StartFileUploadVO startUpload(Resource resource, CreatFileUploadInfoVO creatFileUploadInfoVO) {
        var startFileUploadParam = new StartFileUploadParam<>();
        startFileUploadParam.setFileCode(creatFileUploadInfoVO.getFileCode());
        startFileUploadParam.setFileToKen(creatFileUploadInfoVO.getFileToKen());

        ApiResult<StartFileUploadVO> uploadResult = apiRequest.startFileUploadResource(resource, startFileUploadParam);
        if (!uploadResult.isSuccess()) {
            throw new BusinessException(uploadResult.getMsg());
        }

        return uploadResult.getData();
    }

    private FileObjDTO<T> vo2DTO(StartFileUploadVO uploadVO) {
        FileObjDTO<T> fileObjDTO = new FileObjDTO<>();
        fileObjDTO.setFileCode(uploadVO.getFileCode());
        fileObjDTO.setOriginalName(uploadVO.getOriginalName());
        fileObjDTO.setFileSize(uploadVO.getFileSize());
        fileObjDTO.setSuffix(uploadVO.getSuffix());
        fileObjDTO.setMimeType(uploadVO.getMimeType());
        fileObjDTO.setFileType(FileUploadUtil.obtainFileType(uploadVO.getMimeType()));
        fileObjDTO.setUploadTime(uploadVO.getCreated());
        fileObjDTO.setDownloadPath(uploadVO.getDownloadPath());
        fileObjDTO.setBucketName(uploadVO.getFileDirectoryPath());
        fileObjDTO.setUrl(uploadVO.getUrl());
        fileObjDTO.setFilePath(uploadVO.getFilePath());
        fileObjDTO.setAttribute1(uploadVO.getAttribute1());
        fileObjDTO.setAttribute2(uploadVO.getAttribute2());
        fileObjDTO.setAttribute3(uploadVO.getAttribute3());
        fileObjDTO.setAttribute4(uploadVO.getAttribute4());
        fileObjDTO.setAttribute5(uploadVO.getAttribute5());

        return fileObjDTO;
    }

    private FileObjDTO<T> vo2DTO(FileUploadInfoVO infoVO) {
        FileObjDTO<T> fileObjDTO = new FileObjDTO<>();
        fileObjDTO.setFileCode(infoVO.getFileCode());
        fileObjDTO.setOriginalName(infoVO.getOriginalName());
        fileObjDTO.setFileSize(infoVO.getFileSize());
        fileObjDTO.setSuffix(infoVO.getSuffix());
        fileObjDTO.setMimeType(infoVO.getMimeType());
        fileObjDTO.setFileType(FileUploadUtil.obtainFileType(infoVO.getMimeType()));
        fileObjDTO.setUploadTime(infoVO.getCreated());
        fileObjDTO.setDownloadPath(infoVO.getDownloadPath());
        fileObjDTO.setBucketName(infoVO.getFileDirectoryPath());
        fileObjDTO.setUrl(infoVO.getUrl());
        fileObjDTO.setFilePath(infoVO.getFilePath());
        fileObjDTO.setAttribute1(infoVO.getAttribute1());
        fileObjDTO.setAttribute2(infoVO.getAttribute2());
        fileObjDTO.setAttribute3(infoVO.getAttribute3());
        fileObjDTO.setAttribute4(infoVO.getAttribute4());
        fileObjDTO.setAttribute5(infoVO.getAttribute5());

        return fileObjDTO;
    }
}
