package com.el.coordinator.boot.fsm.config;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.el.coordinator.boot.fsm.common.StorageEnum;
import com.el.coordinator.boot.fsm.config.child.ChildProcessor;
import com.el.coordinator.boot.fsm.config.child.ThumbnailImageProcessor;
import com.el.coordinator.boot.fsm.config.handler.Handlable;
import com.el.coordinator.boot.fsm.config.handler.SimpleFileHandler;
import com.el.coordinator.boot.fsm.config.validator.BlockSuffixValidator;
import com.el.coordinator.boot.fsm.config.validator.FileSizeValidator;
import com.el.coordinator.boot.fsm.config.validator.Validatable;
import com.el.coordinator.boot.fsm.service.FileService;
import com.el.coordinator.boot.fsm.service.exportdata.DataExport;
import com.el.coordinator.boot.fsm.service.impl.FileServiceImpl;
import com.el.coordinator.boot.fsm.service.importdata.DataImport;
import com.el.coordinator.boot.fsm.service.storage.FileStorageService;
import com.el.coordinator.boot.fsm.service.storage.local.LocalFileStorageServiceImpl;
import com.el.coordinator.boot.fsm.service.storage.remote.RemoteFileStorageServiceImpl;
import com.el.coordinator.boot.fsm.support.DataExportServiceFactory;
import com.el.coordinator.boot.fsm.support.DataImportServiceFactory;
import com.el.coordinator.boot.fsm.support.FsmTmplSupport;
import com.el.coordinator.file.api.FsmApiConstant;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateDeserializer;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateSerializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.TrustSelfSignedStrategy;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.ssl.SSLContextBuilder;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.boot.web.client.RestTemplateCustomizer;
import org.springframework.context.annotation.Bean;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.StringRedisSerializer;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.http.converter.xml.MappingJackson2XmlHttpMessageConverter;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * 文件服务配置.
 *
 * @author Kaiser（wang shao）
 * @date 2021-04-15
 */
@Slf4j
public class FileConfiguration<T> {

    private final FileConfigProperties fileConfigProperties;
    private final ElFsmConfigurationCustomizer<T> configurationCustomizer;

    public FileConfiguration(FileConfigProperties fileConfigProperties, ElFsmConfigurationCustomizer<T> configurationCustomizer) {
        this.fileConfigProperties = fileConfigProperties;
        this.configurationCustomizer = configurationCustomizer;
    }

    @Bean
    @ConditionalOnMissingBean(FileStorageService.class)
    public FileStorageService<T> fileStorageService() {
        var storageEnum = fileConfigProperties.getStorageType();
        if (storageEnum == null) {
            throw new IllegalArgumentException("文件服务配置有误，storageType不能为空");
        }

        if (storageEnum == StorageEnum.CUSTOMIZE) {
            throw new IllegalArgumentException("文件服务配置有误，未发现有效的FileStorageService的bean");
        }
        if (storageEnum == StorageEnum.LOCAL) {
            return createLocalStorageService();
        }
        return createRemoteStorageService();
    }

    @Bean
    @ConditionalOnMissingBean(Handlable.class)
    public Handlable<T> simpleFileHandler() {
        return new SimpleFileHandler<>();
    }

    @Bean
    public FileService<T> fileService(FileStorageService<T> fileStorageService, Handlable<T> handlable) {
        List<Validatable<T>> validatableList = createValidatableList();
        List<ChildProcessor<T>> childProcessorList = createChildProcessorList();

        return new FileServiceImpl<>(fileConfigProperties, fileStorageService, validatableList, childProcessorList, handlable);
    }

    @Bean
    public FsmTmplSupport fsmTmplSupport(RedisTemplate<Object, Object> redisTemplate) {
        FileConfigProperties.StorageRemote configProperties = fileConfigProperties.getRemote();
        Assert.notBlank(configProperties.getServerUrl(), "文件服务配置失败，serverUrl不能为空");
        var clientHttpRequestFactory = getClientHttpRequestFactory(configProperties);
        var restTemplate = new RestTemplateBuilder()
                .requestFactory(() -> clientHttpRequestFactory)
                .rootUri(configProperties.getServerUrl() + FsmApiConstant.DEFAULT_FSM_CONTEXT_PATH)
                .customizers(restTemplateCustomizer())
                .build();

        return new FsmTmplSupport(restTemplate, redisTemplate);
    }

    @Bean
    public DataImportServiceFactory dataImportServiceFactory(List<DataImport<?>> dataImportList) {
        return new DataImportServiceFactory(dataImportList);
    }

    @Bean
    public DataExportServiceFactory dataExportServiceFactory(List<DataExport<?, ?>> dataExportList) {
        return new DataExportServiceFactory(dataExportList);
    }

    @Bean
    @ConditionalOnMissingBean(name = "redisTemplate")
    public RedisTemplate<Object, Object> redisTemplate(RedisConnectionFactory redisConnectionFactory) {
        RedisTemplate<Object, Object> template = new RedisTemplate<>();
        template.setConnectionFactory(redisConnectionFactory);
        template.setKeySerializer(new StringRedisSerializer());
        return template;
    }

    @Bean("yst_el_taskExecutor")
    @ConditionalOnMissingBean(name = "yst_el_taskExecutor")
    public ThreadPoolTaskExecutor taskExecutor() {
        var config = fileConfigProperties.getThread();

        var executor = new ThreadPoolTaskExecutor();
        executor.setCorePoolSize(config.getCoreThread());
        executor.setMaxPoolSize(config.getMaxThread());
        executor.setQueueCapacity(config.getQueueSize());
        executor.setKeepAliveSeconds(300);
        executor.setWaitForTasksToCompleteOnShutdown(false);
        executor.setThreadNamePrefix(config.getThreadPrefix());
        executor.setRejectedExecutionHandler(new ThreadPoolExecutor.AbortPolicy());
        executor.initialize();
        return executor;
    }

    private ClientHttpRequestFactory getClientHttpRequestFactory(FileConfigProperties.StorageRemote configProperties) {
        SSLConnectionSocketFactory sslConnectionSocketFactory = null;

        try {
            SSLContextBuilder sslContextBuilder = new SSLContextBuilder();
            sslContextBuilder.loadTrustMaterial(null, new TrustSelfSignedStrategy());
            sslConnectionSocketFactory = new SSLConnectionSocketFactory(sslContextBuilder.build(), NoopHostnameVerifier.INSTANCE);
        } catch (Exception e) {
            e.printStackTrace();
        }

        HttpClientBuilder httpClientBuilder = HttpClients.custom().disableContentCompression();
        httpClientBuilder.setSSLSocketFactory(sslConnectionSocketFactory);

        var requestFactory = new HttpComponentsClientHttpRequestFactory();
        requestFactory.setHttpClient(httpClientBuilder.build());
        requestFactory.setConnectTimeout((int) configProperties.getServerConnectTimeout().toMillis());
        requestFactory.setReadTimeout((int) configProperties.getServerReadTimeout().toMillis());
        return requestFactory;
    }

    private RestTemplateCustomizer restTemplateCustomizer() {
        var objectMapper = objectMapper();
        return template -> {
            MappingJackson2HttpMessageConverter jsonConvert = null;

            var xmlConvertIndex = -1;
            int index = 0;
            for (var convert : template.getMessageConverters()) {
                if (convert instanceof MappingJackson2XmlHttpMessageConverter) {
                    xmlConvertIndex = index;
                }
                if (convert instanceof MappingJackson2HttpMessageConverter) {
                    jsonConvert = (MappingJackson2HttpMessageConverter) convert;
                }
                index++;
            }
            if (jsonConvert == null) {
                jsonConvert = new MappingJackson2HttpMessageConverter(objectMapper);
                template.getMessageConverters().add(jsonConvert);
            } else {
                jsonConvert.setObjectMapper(objectMapper);
            }

            if (xmlConvertIndex >= 0) {
                template.getMessageConverters().remove(jsonConvert);
                template.getMessageConverters().add(xmlConvertIndex - 1, jsonConvert);
            }
        };
    }

    private ObjectMapper objectMapper() {
        var objectMapper = new ObjectMapper();

        var javaTimeModule = new JavaTimeModule();
        javaTimeModule.addDeserializer(LocalDateTime.class, new LocalDateTimeDeserializer(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));
        javaTimeModule.addSerializer(LocalDateTime.class, new LocalDateTimeSerializer(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));
        javaTimeModule.addDeserializer(LocalDate.class, new LocalDateDeserializer(DateTimeFormatter.ofPattern("yyyy-MM-dd")));
        javaTimeModule.addSerializer(LocalDate.class, new LocalDateSerializer(DateTimeFormatter.ofPattern("yyyy-MM-dd")));
        objectMapper.registerModule(javaTimeModule);

        objectMapper.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
        return objectMapper;
    }

    private FileStorageService<T> createLocalStorageService() {
        return new LocalFileStorageServiceImpl<>(fileConfigProperties.getLocal());
    }

    private FileStorageService<T> createRemoteStorageService() {
        FileConfigProperties.StorageRemote configProperties = fileConfigProperties.getRemote();
        Assert.notBlank(configProperties.getServerUrl(), "文件服务配置失败，serverUrl不能为空");
        Assert.notBlank(configProperties.getProjectFlag(), "文件服务配置失败，projectFlag不能为空");

        var clientHttpRequestFactory = getClientHttpRequestFactory(configProperties);
        var restTemplate = new RestTemplateBuilder()
                .requestFactory(() -> clientHttpRequestFactory)
                .customizers(restTemplateCustomizer())
                .build();
        return new RemoteFileStorageServiceImpl<>(fileConfigProperties.getRemote(), restTemplate, objectMapper());
    }

    private List<Validatable<T>> createValidatableList() {
        List<Validatable<T>> validatableList = ObjectUtil.defaultIfNull(configurationCustomizer.getValidators(), Collections.emptyList());
        if (CollUtil.isEmpty(validatableList)) {
            validatableList = new ArrayList<>(8);
        }
        validatableList.add(new FileSizeValidator<>(fileConfigProperties));
        validatableList.add(new BlockSuffixValidator<>(fileConfigProperties));

        return validatableList;
    }

    private List<ChildProcessor<T>> createChildProcessorList() {
        List<ChildProcessor<T>> processorList = ObjectUtil.defaultIfNull(configurationCustomizer.getChildProcessors(), Collections.emptyList());
        if (CollUtil.isEmpty(processorList)) {
            processorList = new ArrayList<>(8);
        }
        processorList.add(new ThumbnailImageProcessor<>());

        return processorList;
    }
}
