package com.el.coordinator.boot.fsm.config.child;

import cn.hutool.core.img.Img;
import cn.hutool.core.util.StrUtil;
import com.el.coordinator.boot.fsm.common.ConstantsFile;
import com.el.coordinator.boot.fsm.common.FileTypeEnum;
import com.el.coordinator.boot.fsm.common.UploadFileParam;
import com.el.coordinator.boot.fsm.model.dto.FileObjDTO;
import com.el.coordinator.boot.fsm.model.dto.FileUploadDTO;
import com.el.coordinator.boot.fsm.util.FileUploadUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.Resource;
import org.springframework.util.unit.DataSize;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Collections;
import java.util.function.Function;

/**
 * 图片的缩略图.
 *
 * @author Kaiser（wang shao）
 * @date 2021-04-19
 */
@Slf4j
public class ThumbnailImageProcessor<T> implements ChildProcessor<T> {

    /**
     * 缩略图的最大
     */
    private final DataSize SIZE_MAX = DataSize.ofKilobytes(50);

    private final String CHILD_FLAG = ConstantsFile.PARAM_FILE_CHILD_IMG_THUMBNAIL;

    @Override
    public boolean accept(String childType) {
        return StrUtil.equals(childType, CHILD_FLAG);
    }

    @Override
    public void process(UploadFileParam<T> uploadFileParam, FileObjDTO<T> parentFile, Function<FileUploadDTO<T>, FileObjDTO<T>> uploadMethod) {
        if (uploadFileParam.getFileType() == null || uploadFileParam.getFileType() != FileTypeEnum.IMAGE) {
            return;
        }

        try {
            Resource resource = generateThumbnail(uploadFileParam);
            uploadMethod.apply(buildFileUploadDto(resource, parentFile));
        } catch (IOException e) {
            log.error("生成图片缩略图失败：", e);
            throw new RuntimeException("生成图片缩略图失败");
        }
    }

    private FileUploadDTO<T> buildFileUploadDto(Resource resource, FileObjDTO<T> parentFile) throws IOException {
        FileUploadDTO<T> fileUploadDTO = new FileUploadDTO<>();
        fileUploadDTO.setFileParam((UploadFileParam<T>) FileUploadUtil.resource2UploadFileParam(resource, Collections.emptyMap()));
        fileUploadDTO.setParentFileCode(parentFile.getFileCode());
        fileUploadDTO.setChildFlag(CHILD_FLAG);
        fileUploadDTO.setAttribute1(parentFile.getAttribute1());
        fileUploadDTO.setAttribute2(parentFile.getAttribute2());
        fileUploadDTO.setAttribute3(parentFile.getAttribute3());
        fileUploadDTO.setAttribute4(parentFile.getAttribute4());
        fileUploadDTO.setAttribute5(parentFile.getAttribute5());
        return fileUploadDTO;

    }

    private Resource generateThumbnail(UploadFileParam<T> uploadFileParam) throws IOException {
        double max = SIZE_MAX.toBytes() * 1.0;
        long size = uploadFileParam.getFileSize().toBytes();
        float quality = (float) (max / size);
        if (quality >= 1) {
            // 无需生成
            return uploadFileParam.getUploadFile();
        }
        Resource thumbnail = uploadFileParam.getUploadFile();
        var image = Img.from(thumbnail.getInputStream()).getImg();

        ByteArrayOutputStream outputStream = null;
        float scale = 1;
        while (true) {
            outputStream = new ByteArrayOutputStream();
            Img.from(image)
                    .scale(scale)
                    .setQuality(quality)
                    .write(outputStream);

            size = outputStream.size();
            quality = (float) (max / size);
            scale = getScale(size);
            if (quality >= 1) {
                // 已达标，无需再压缩
                break;
            }

            // 仍需要压缩
            image = Img.from(new ByteArrayInputStream(outputStream.toByteArray())).getImg();
        }

        return new ByteArrayResource(outputStream.toByteArray()) {
            @Override
            public String getFilename() {
                return thumbnail.getFilename();
            }
        };
    }

    /**
     * 自动调节精度(经验数值)
     *
     * @param size 源图片大小
     * @return 图片压缩质量比
     */
    private static float getScale(long size) {
        float accuracy;
        if (size < 900) {
            accuracy = 0.85F;
        } else if (size < 2047) {
            accuracy = 0.6F;
        } else if (size < 3275) {
            accuracy = 0.44F;
        } else {
            accuracy = 0.4F;
        }
        return accuracy;
    }
}
