package com.el.coordinator.boot.fsm.controller;

import com.el.coordinator.boot.fsm.model.vo.ExportResultRespVO;
import com.el.coordinator.boot.fsm.model.vo.ImportRateRespVO;
import com.el.coordinator.boot.fsm.model.vo.ImportResultRespVO;
import com.el.coordinator.boot.fsm.service.DataChannelService;
import com.el.coordinator.core.common.api.ApiResult;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.github.xiaoymin.knife4j.annotations.ApiSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.InputStreamResource;
import org.springframework.http.HttpEntity;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;
import springfox.documentation.annotations.ApiIgnore;

import java.util.Map;

/**
 * 数据导入导出
 *
 * @author Kaiser（wang shao）
 * @date 2021/6/1
 */
@Api(tags = "数据导入导出服务", hidden = true)
@ApiIgnore
@ApiSupport(author = "wang shao", order = 2)
@RestController
@RequestMapping(value = "/com/import", produces = MediaType.APPLICATION_JSON_VALUE)
public class DataChannelController {

    @Autowired
    private DataChannelService dataChannelService;

    /**
     * 下载模板文件
     *
     * @param tmplCode 模板编号
     * @return 文件流
     * @download
     */
    @GetMapping(value = "/{tmplCode}/tmpl", produces = MediaType.APPLICATION_OCTET_STREAM_VALUE)
    @ApiOperation(value = "下载模板文件")
    @ApiOperationSupport(order = 1)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "tmplCode", value = "模板编号", required = true, paramType = "path"),
    })
    public ResponseEntity<InputStreamResource> downloadTmpl(@PathVariable String tmplCode) {
        return dataChannelService.downloadByCode(tmplCode);
    }

    /**
     * 根据模板导入数据
     *
     * @param tmplCode 模板编号
     * @param file     数据文件
     * @return 导入结果
     */
    @PostMapping(value = "/{tmplCode}/data")
    @ApiOperation(value = "根据模板导入数据")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "tmplCode", value = "模板编号", required = true, paramType = "path"),
            @ApiImplicitParam(name = "file", value = "数据文件", required = true),
    })
    public ApiResult<ImportResultRespVO> importData(@PathVariable String tmplCode, MultipartFile file) {
        return dataChannelService.importData(tmplCode, file);
    }

    /**
     * 根据模板导出数据
     *
     * @param tmplCode   模板编号
     * @param queryParam 查询参数
     * @return 导出结果
     */
    @PostMapping(value = "/{tmplCode}/export")
    @ApiOperation(value = "根据模板导出数据")
    @ApiOperationSupport(order = 3)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "tmplCode", value = "模板编号", required = true, paramType = "path"),
            @ApiImplicitParam(name = "queryParam", value = "查询参数", paramType = "body", dataType = "map"),
    })
    public ApiResult<ExportResultRespVO> exportData(@PathVariable String tmplCode, @RequestBody(required = false) Object queryParam) {
        return dataChannelService.exportData(tmplCode, (Map<String, Object>) queryParam);
    }

    /**
     * 下载导出的数据文件
     *
     * @param recordId 导出记录ID
     * @return 数据文件流
     * @download
     */
    @GetMapping(value = "/export/{recordId}/file", produces = MediaType.APPLICATION_OCTET_STREAM_VALUE)
    @ApiOperation(value = "下载导出的数据文件")
    @ApiOperationSupport(order = 4)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "recordId", value = "导出记录ID", required = true),
    })
    public HttpEntity<StreamingResponseBody> downloadExportFile(@PathVariable Long recordId) {
        return dataChannelService.downloadExportFile(recordId);
    }

    /**
     * 查询导入（或导出）进度
     *
     * @param recordId 导入(导出)记录ID
     * @return 进度
     */
    @GetMapping(value = "/{recordId}/rate")
    @ApiOperation(value = "查询导入（或导出）进度")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "recordId", value = "记录标识", required = true),
    })
    public ApiResult<ImportRateRespVO> getRate(@PathVariable Long recordId) {
        return dataChannelService.getRate(recordId);
    }
}
