package com.el.coordinator.boot.fsm.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.exceptions.ExceptionUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.el.coordinator.boot.fsm.convert.TmplImportConvert;
import com.el.coordinator.boot.fsm.model.bo.ExportStrategyParam;
import com.el.coordinator.boot.fsm.model.vo.ExportResultRespVO;
import com.el.coordinator.boot.fsm.model.vo.FileObjRespVO;
import com.el.coordinator.boot.fsm.model.vo.ImportRateRespVO;
import com.el.coordinator.boot.fsm.model.vo.ImportResultRespVO;
import com.el.coordinator.boot.fsm.service.DataChannelService;
import com.el.coordinator.boot.fsm.service.FileService;
import com.el.coordinator.boot.fsm.service.FileUserService;
import com.el.coordinator.boot.fsm.service.exportdata.DataExport;
import com.el.coordinator.boot.fsm.service.impl.exportstrategy.ExportStrategyDelegate;
import com.el.coordinator.boot.fsm.service.importdata.DataImport;
import com.el.coordinator.boot.fsm.support.DataExportServiceFactory;
import com.el.coordinator.boot.fsm.support.DataImportServiceFactory;
import com.el.coordinator.boot.fsm.support.FsmTmplSupport;
import com.el.coordinator.boot.fsm.util.FileUploadUtil;
import com.el.coordinator.boot.fsm.util.excel.ExcelImportUtil;
import com.el.coordinator.core.common.api.ApiResult;
import com.el.coordinator.core.common.constant.ConstantFsm;
import com.el.coordinator.core.common.exception.BusinessException;
import com.el.coordinator.core.common.jpa.vo.PagingVO;
import com.el.coordinator.file.business.dto.ImportRateDTO;
import com.el.coordinator.file.business.dto.TmplDTO;
import com.el.coordinator.file.business.param.ImportResultDTO;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.io.InputStreamResource;
import org.springframework.http.HttpEntity;
import org.springframework.http.ResponseEntity;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import javax.annotation.PostConstruct;
import java.io.File;
import java.io.Serializable;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/6/3
 */
@Service
@Slf4j
public class DataImportServiceImpl implements DataChannelService {

    @Autowired
    private FsmTmplSupport fsmTmplSupport;
    @Autowired
    private DataImportServiceFactory dataImportServiceFactory;
    @Autowired
    private DataExportServiceFactory dataExportServiceFactory;
    @Autowired(required = false)
    private FileUserService fileUserService;
    @Autowired
    private FileService fileService;
    @Autowired
    @Qualifier("yst_el_taskExecutor")
    private ThreadPoolTaskExecutor taskExecutor;
    @Autowired
    private ExportStrategyDelegate exportStrategyDelegate;

    // 临时文件夹
    private File tempDir = null;
    private static final ObjectMapper OBJECT_MAPPER = FileUploadUtil.createObjectMapper();

//    @PostConstruct
    private void init() {
        // 处理系统异常停止的记录状态
        CompletableFuture.runAsync(() -> {
            List<Long> importIds = fsmTmplSupport.getUnFinished();
            if (CollUtil.isNotEmpty(importIds)) {
                ImportResultDTO resultDTO = null;
                ImportRateDTO rateDTO = null;
                for (Long id : importIds) {
                    rateDTO = fsmTmplSupport.getImportRateFromCache(id);
                    resultDTO = ImportResultDTO.builder()
                            .success(false)
                            .failMsg("系统异常停止")
                            .build();
                    if (rateDTO != null) {
                        resultDTO.setNumSuc(rateDTO.getCount());
                    }
                    fsmTmplSupport.updateImportResult(id, resultDTO);
                }
            }
        }, taskExecutor).exceptionally(throwable -> {
            log.warn("处理未导入结束的失败", throwable);
            return null;
        });

        initTempDir();
    }

    private void initTempDir() {
        tempDir = new File(System.getProperty("java.io.tmpdir"));
        if (tempDir.exists()) {
            return;
        }
        if (!tempDir.mkdirs()) {
            throw new IllegalArgumentException("创建临时文件夹失败：" + tempDir.getAbsolutePath());
        }
    }

    @Override
    public ResponseEntity<InputStreamResource> downloadByCode(String code) {
        return fsmTmplSupport.downloadByCode(code);
    }

    @Override
    public ApiResult<ImportResultRespVO> importData(String code, MultipartFile dataFile) {
        // 导入前校验
        var tmplDTO = fsmTmplSupport.getTmplByCode(code);
        String msg = validateBeforeImport(tmplDTO);
        if (msg != null) {
            return ApiResult.fail(msg);
        }

        // 保存导入记录
        Long importId = null;
        try {
            importId = saveRecord(tmplDTO, dataFile, null);
        } catch (Exception exception) {
            afterImport(tmplDTO, null, null, ExceptionUtil.getRootCause(exception).getMessage());
            return ApiResult.fail("导入失败，文件服务器异常");
        }

        // 获取数据导入服务
        var dataImportService = dataImportServiceFactory.getDataImportService(code);

        // 开始导入
        ImportResultRespVO respVO = null;
        try {
            respVO = startImport(importId, tmplDTO, dataFile, dataImportService);
        } catch (Exception exception) {
            log.error("导入失败", exception);
            afterImport(tmplDTO, importId, 0L, ExceptionUtil.getRootCause(exception).getMessage());
            return ApiResult.fail("导入失败" + (exception instanceof BusinessException ? "，" + exception.getMessage() : ""));
        }
        if (Boolean.TRUE.equals(respVO.getSync())) {
            afterImport(tmplDTO, importId, (long) ObjectUtil.defaultIfNull(respVO.getSyncResult().getNumSuccess(), 0), String.join("；", respVO.getSyncResult().getFailRecords()));
        }
        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<ExportResultRespVO> exportData(String code, Map<String, Object> queryParam) {
        // 导入前校验
        var tmplDTO = fsmTmplSupport.getTmplByCode(code);
        String msg = validateBeforeImport(tmplDTO);
        if (msg != null) {
            return ApiResult.fail(msg);
        }

        // 保存导出记录
        Long importId = null;
        try {
            importId = saveRecord(tmplDTO, null, queryParam);
        } catch (Exception exception) {
            afterImport(tmplDTO, null, null, ExceptionUtil.getRootCause(exception).getMessage());
            return ApiResult.fail("导出失败，文件服务器异常");
        }

        // 获取数据导出服务
        var dataExportService = dataExportServiceFactory.getDataExportService(code);

        // 开始导出
        AnalyseResult result;
        try {
            result = startExport(importId, tmplDTO, queryParam, dataExportService);
        } catch (Exception exception) {
            log.error("导出失败", exception);
            afterImport(tmplDTO, importId, 0L, ExceptionUtil.getRootCause(exception).getMessage());
            return ApiResult.fail("导出失败" + (exception instanceof BusinessException ? "，" + exception.getMessage() : ""));
        }

        return ApiResult.ok(new ExportResultRespVO(result.getSync(), importId, result.getTotal()));
    }

    @Override
    public HttpEntity<StreamingResponseBody> downloadExportFile(Long id) {
        String code = FileUploadUtil.retry(() -> fsmTmplSupport.getRecordFileCode(id), StrUtil::isNotBlank, Duration.ofMinutes(1));
        if (StrUtil.isBlank(code)) {
            return ResponseEntity.badRequest().build();
        }

        return fileService.download(code, null);
    }

    @Override
    public ApiResult<ImportRateRespVO> getRate(Long id) {
        ImportRateDTO rateDTO = null;
        try {
            rateDTO = fsmTmplSupport.getImportRateFromCache(id);
            if (rateDTO == null || ObjectUtil.equals(rateDTO.getTotal(), rateDTO.getCount())) {
                rateDTO = fsmTmplSupport.getImportRate(id);
            }
        } catch (Exception exception) {
            log.error("查询导入进度失败", exception);
            return ApiResult.fail("查询导入进度失败");
        }

        if (rateDTO == null) {
            return ApiResult.fail("导入记录不存在");
        }
        var respVo = TmplImportConvert.INSTANCE.dto2Vo(rateDTO);
        respVo.setRate(BigDecimal.valueOf(rateDTO.getCount() * 1.0 / rateDTO.getTotal() * 100).setScale(0, RoundingMode.DOWN) + "%");

        return ApiResult.ok(respVo);
    }

    private Long saveRecord(TmplDTO tmplDTO, MultipartFile dataFile, Map<String, Object> param) {
        if (fileUserService == null) {
            return fsmTmplSupport.saveRecord(tmplDTO.getCode(), dataFile, null, param);
        }
        return fsmTmplSupport.saveRecord(tmplDTO.getCode(), dataFile, fileUserService.currentUser(), param);
    }

    private boolean updateLimiter(TmplDTO tmplDTO, boolean add) {
        if (tmplDTO.getConcurrentLimit() == null || tmplDTO.getConcurrentLimit() == -1) {
            // 不限流
            return true;
        }
        return FsmTmplSupport.updateLimiter(tmplDTO.getCode(), tmplDTO.getConcurrentLimit().toString(), 1, add);
    }

    private String validateBeforeImport(TmplDTO tmplDTO) {
        if (tmplDTO == null || Boolean.FALSE.equals(tmplDTO.getEnabled())) {
            return "数据服务不存在或未启用";
        }
        if (CollectionUtil.isEmpty(tmplDTO.getAttributes())) {
            return "模板无效";
        }

        boolean export = Boolean.TRUE.equals(tmplDTO.getExport());
        if (export) {
            if (!dataExportServiceFactory.isSupport(tmplDTO.getCode())) {
                return "未发现有效的数据导出服务";
            }
        } else {
            if (!dataImportServiceFactory.isSupport(tmplDTO.getCode())) {
                return "未发现有效的数据导入服务";
            }
        }

        // 限流设置
        if (!updateLimiter(tmplDTO, true)) {
            return "当前访问用户过多，请稍后再试";
        }

        return null;
    }

    private void afterImport(TmplDTO tmplDTO, Long importId, Long numSuc, String msg) {
        afterImport(tmplDTO, importId, numSuc, msg, null);
    }

    private void afterImport(TmplDTO tmplDTO, Long importId, Long numSuc, String msg, File file) {
        // 更新导入结果
        if (importId != null) {
            String fileCode = uploadImportFile(file);
            var importResultDTO = ImportResultDTO.builder()
                    .success(StrUtil.isBlank(msg))
                    .numSuc(numSuc)
                    .failMsg(msg)
                    .fileCode(fileCode)
                    .build();
            fsmTmplSupport.updateImportResult(importId, importResultDTO);
        } else {
            log.error("更新导入导出记录结果失败：{}", msg);
        }

        // 更新限流信息
        updateLimiter(tmplDTO, false);
    }

    private String uploadImportFile(File file) {
        if (file != null) {
            ApiResult<FileObjRespVO> uploadResult = fileService.upload(file);
            // 删除临时文件
            file.delete();
            file.getParentFile().delete();
            if (uploadResult.isSuccess()) {
                return uploadResult.getData().getFileCode();
            } else {
                log.error("上传导入导出结果文件失败：{}", uploadResult);
            }
        }
        return null;
    }

    private List<?> analyseData(TmplDTO tmplDTO, MultipartFile dataFile, DataImportServiceFactory.ServiceMetaData dataImportService) {
        try {
            return ExcelImportUtil.instance(dataFile.getInputStream())
                    .headRow(tmplDTO.getHeadRow())
                    .dataType(dataImportService.getDataType(), tmplDTO.getAttributes().get(tmplDTO.getFieldTypeRow() - 1))
                    .readAllSync();
        } catch (Exception e) {
            log.error("解析导入数据失败", e);
            throw new BusinessException("解析导入数据失败");
        }
    }

    private AnalyseResult startExport(Long importId, TmplDTO tmplDTO, Map<String, Object> queryParam, DataExportServiceFactory.ServiceMetaData dataExportService) {
        var exportService = dataExportService.getDataExport();
        var param = convertParam(queryParam, dataExportService.getParamType());
        var queryResult = exportService.execute(param, 1, obtainPageSize(dataExportService.getDataExport()));
        if (queryResult == null || queryResult.getTotal() == 0) {
            throw new BusinessException("没有符合条件的数据");
        }

        int limit = ObjectUtil.defaultIfNull(tmplDTO.getDataLimitPer(), ConstantFsm.UN_LIMIT);
        if (limit != -1 && queryResult.getTotal() > limit) {
            throw new BusinessException("每次最多允许导出" + limit + "条");
        }
        // 更新导出总数量
        CompletableFuture.runAsync(() -> fsmTmplSupport.updateImportNum(importId, queryResult.getTotal()), taskExecutor);

        // 判断是否需要异步
        var sync = tmplDTO.getAsyncThreshold() == null || tmplDTO.getAsyncThreshold() == -1 || queryResult.getTotal() <= tmplDTO.getAsyncThreshold();

        if (sync) {
            exportSync(importId, tmplDTO, queryResult, param, exportService);

            return new AnalyseResult(queryResult.getTotal(), true);
        }
        exportAsync(importId, tmplDTO, queryResult, param, exportService);

        return new AnalyseResult(queryResult.getTotal(), false);
    }

    private void exportAsync(Long importId, TmplDTO tmplDTO, PagingVO<Serializable> pageData, Serializable queryParam,
                             DataExport<Serializable, Serializable> dataExport) {
        // 设置当前的用户认证
        var authentication = SecurityContextHolder.getContext().getAuthentication();

        CompletableFuture.supplyAsync(() -> {
                    if (authentication != null) {
                        SecurityContextHolder.getContext().setAuthentication(authentication);
                    }
                    return write2Excel(false, importId, tmplDTO, pageData, queryParam, dataExport);
                }, taskExecutor)
                .whenComplete((ret, throwable) -> {
                    fsmTmplSupport.removeImportRate(importId);
                    if (throwable == null) {
                        afterImport(tmplDTO, importId, ret, null);
                    } else {
                        afterImport(tmplDTO, importId, 0L, ExceptionUtil.getRootCause(throwable).getMessage());
                        log.error("导出数据时出现异常：", throwable);
                    }
                });
    }

    private int obtainPageSize(DataExport<Serializable, Serializable> dataExportService) {
        Integer pageSize = dataExportService.pageSize();
        if (pageSize == null || pageSize < 1) {
            // 默认页大小
            return 500;
        }

        return pageSize;
    }

    private int obtainStepSize(DataImport<Serializable> dataImport) {
        Integer stepSize = dataImport.stepSize();
        if (stepSize == null || stepSize < 1) {
            // 默认页大小
            return 10;
        }

        return stepSize;
    }

    private void exportSync(Long importId, TmplDTO tmplDTO,
                            PagingVO<Serializable> pageData, Serializable queryParam, DataExport<Serializable, Serializable> dataExport) {
        Long num = write2Excel(true, importId, tmplDTO, pageData, queryParam, dataExport);
        afterImport(tmplDTO, importId, num, null);
    }

    private Long write2Excel(boolean sync, Long importId, TmplDTO tmplDTO, PagingVO<Serializable> pageData,
                             Serializable queryParam, DataExport<Serializable, Serializable> dataExport) {
        var param = new ExportStrategyParam()
                .setSync(sync)
                .setImportId(importId)
                .setTmplDTO(tmplDTO)
                .setFirstPageData(pageData)
                .setQueryParam(queryParam)
                .setDataExport(dataExport);
        return (long) exportStrategyDelegate.export(param);
    }

    private <E extends Serializable> E convertParam(Map<String, Object> queryParam, Class<E> paramType) {
        try {
            return OBJECT_MAPPER.convertValue(queryParam, paramType);
        } catch (IllegalArgumentException e) {
            throw new BusinessException("转换查询参数失败，请检查参数格式", e);
        }
    }

    private ImportResultRespVO startImport(Long importId, TmplDTO tmplDTO, MultipartFile dataFile,
                                           DataImportServiceFactory.ServiceMetaData dataImportService) {
        // 判断是否需要异步
        List dataList = analyseData(tmplDTO, dataFile, dataImportService);
        var total = dataList.size();

        if (total == 0) {
            throw new BusinessException("导入数据为空");
        }
        if (tmplDTO.getDataLimitPer() != -1 && total > tmplDTO.getDataLimitPer()) {
            throw new BusinessException("每次最多允许导入" + tmplDTO.getDataLimitPer() + "条");
        }

        // 更新导入总数量
        fsmTmplSupport.updateImportNum(importId, (long) total);

        var sync = tmplDTO.getAsyncThreshold() == null || tmplDTO.getAsyncThreshold() == -1 || total <= tmplDTO.getAsyncThreshold();
        if (sync) {
            // 需要同步处理
            var syncResult = dataImportService.getDataImport().execute(dataList, tmplDTO.getHeadRow() + 1);
            return ImportResultRespVO.builder().sync(true).syncResult(syncResult).build();
        }

        // 需要异步处理
        CompletableFuture.supplyAsync(() -> importAsync(tmplDTO, importId, dataImportService.getDataImport(), dataList), taskExecutor)
                .whenComplete((ret, throwable) -> {
                    fsmTmplSupport.removeImportRate(importId);

                    if (throwable == null) {
                        // 导入成功
                        afterImport(tmplDTO, importId, (long) ret.getNumSuccess(), String.join("；", ret.getFailRecords()));
                    } else {
                        // 导入失败
                        String failMsg = ret != null && ret.getFailRecords() != null ? String.join("；", ret.getFailRecords()) : "";

                        afterImport(tmplDTO, importId, 0L, failMsg + "；" + ExceptionUtil.getRootCause(throwable).getMessage());
                        log.error("导入数据时出现异常：", throwable);
                    }
                });
        return ImportResultRespVO.builder().sync(false).asyncResult(ImportResultRespVO.AsyncResult.builder().importId(importId).build()).build();
    }

    private ImportResultRespVO.SyncResult importAsync(TmplDTO tmplDTO, Long importId, DataImport<Serializable> dataImportService, List<Serializable> dataList) {
        int total = dataList.size();
        var size = obtainStepSize(dataImportService);
        var start = 0;
        var numSuc = 0;
        ImportResultRespVO.SyncResult tempResult = null;
        List<String> fail = new ArrayList<>(1024);
        for (var i = 0; ; i++) {
            start = i * size;

            try {
                tempResult = dataImportService.execute(dataList.subList(start, Math.min(start + size, total)),
                        tmplDTO.getHeadRow() + 1 + start);
            } catch (Exception exception) {
                log.error("导入失败", exception);
                fail.add(String.format("total：%s，start:%s，exception:%s", total, start, exception.getMessage()));
            }
            if (tempResult != null) {
                numSuc += ObjectUtil.defaultIfNull(tempResult.getNumSuccess(), 0);
                if (CollUtil.isNotEmpty(tempResult.getFailRecords())) {
                    fail.addAll(tempResult.getFailRecords());
                }
            }

            var finish = start + size >= total;
            fsmTmplSupport.storeImportRate(importId, ImportRateDTO.builder()
                    .finish(finish)
                    .total((long) total)
                    .count((long) numSuc)
                    .tmplCode(dataImportService.getTmplCode())
                    .build());
            if (finish) {
                break;
            }
        }

        return ImportResultRespVO.SyncResult.builder().total(total).numSuccess(numSuc).failRecords(fail).build();
    }

    @AllArgsConstructor
    @Data
    static class AnalyseResult {
        /**
         * 总记录数
         */
        private Long total;

        /**
         * 是否同步
         */
        private Boolean sync;
    }
}
