package com.el.coordinator.boot.fsm.service.impl.exportstrategy;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.el.coordinator.boot.fsm.model.bo.ExportStrategyParam;
import com.el.coordinator.boot.fsm.service.FileService;
import com.el.coordinator.boot.fsm.support.FsmTmplSupport;
import com.el.coordinator.file.business.dto.ImportRateDTO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StopWatch;

import java.util.List;

/**
 * 超出sheet限制后，开启新的sheet.
 *
 * @author Kaiser（wang shao）
 * @date 2022/11/28
 */
class NewSheetExportStrategy extends BaseExportStrategy {
    private static final Logger LOG = LoggerFactory.getLogger(NewSheetExportStrategy.class);

    public NewSheetExportStrategy(FileService fileService, FsmTmplSupport fsmTmplSupport) {
        super(fileService, fsmTmplSupport);
    }

    @Override
    public long export(ExportStrategyParam param) {
        int sheetLimit = sheetLimit(param.getTmplDTO());
        int page = 2;
        var pageSize = super.obtainPageSize(param.getDataExport());
        long count = 0;
        int sheetIndex = 0;
        int rowCount = 0;

        var excelFile = createExportFile(param);
        StopWatch watch = new StopWatch();
        watch.start();

        var excelWriter = EasyExcel.write(excelFile).build();
        WriteSheet sheet = EasyExcel.writerSheet(sheetIndex).sheetName("Sheet " + (sheetIndex + 1)).build();
        // 头部
        var headers = obtainHeaders(param.getTmplDTO());
        excelWriter.write(headers, sheet);

        // 开始写数据
        var pageData = param.getFirstPageData();
        List<List<String>> writeDataList;
        var attributes = obtainAttributes(param.getTmplDTO());

        while (pageData.isNotEmpty()) {
            if (rowCount == sheetLimit) {
                // 上一页已满
                sheetIndex++;
                rowCount = 0;
                sheet = EasyExcel.writerSheet(sheetIndex).sheetName("Sheet " + (sheetIndex + 1)).build();
                excelWriter.write(headers, sheet);
            }

            var tempDataList = pageData.getRecords();
            if (rowCount + pageData.size() > sheetLimit) {
                // 超出限制了，则将超出部分写入新的sheet
                int start = 0;
                int end = start + (sheetLimit - rowCount);
                tempDataList = pageData.getRecords().subList(start, end);

                // 新的sheet
                while (end < pageData.size()) {
                    writeDataList = convertExportData(tempDataList, attributes);
                    excelWriter.write(writeDataList, sheet);

                    int newEnd = end + (sheetLimit - rowCount);
                    sheetIndex++;
                    sheet = EasyExcel.writerSheet(sheetIndex).sheetName("Sheet " + (sheetIndex + 1)).build();
                    excelWriter.write(headers, sheet);

                    tempDataList = pageData.getRecords().subList(end, Math.min(pageData.size(), newEnd));
                    rowCount = 0;
                    end = newEnd;
                }
            }
            rowCount += tempDataList.size();
            count += pageData.size();
            writeDataList = convertExportData(tempDataList, attributes);
            excelWriter.write(writeDataList, sheet);

            // 少于页大小，则说明没有数据了
            var isFinish = pageData.size() < pageSize || pageData.getTotal() == count;
            if (!param.isSync()) {
                // 异步时更新进度
                super.updateRate(param.getImportId(), ImportRateDTO.builder().finish(isFinish).total(pageData.getTotal()).count(count).tmplCode(param.getTmplDTO().getCode()).build());
            }
            if (isFinish) {
                break;
            }

            pageData = param.getDataExport().execute(param.getQueryParam(), page, pageSize);
            page++;
        }
        excelWriter.finish();
        watch.stop();

        // 更新导出结果
        updateExportFinish(param, count, "导出成功", excelFile);
        LOG.info(param.getImportId() + "导出结束耗时：" + watch.getTotalTimeMillis());
        excelFile.deleteOnExit();

        return count;
    }
}
