package com.el.coordinator.boot.fsm.service.storage;


import com.el.coordinator.boot.fsm.model.dto.FileObjDTO;
import com.el.coordinator.boot.fsm.model.dto.FileUploadDTO;
import com.el.coordinator.core.common.api.ApiResult;
import com.el.coordinator.file.parameter.FilePackageParam;
import com.el.coordinator.file.vo.FileChunkReqVO;
import com.el.coordinator.file.vo.FileChunkSaveVO;
import com.el.coordinator.file.vo.PackageResultVO;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpEntity;
import org.springframework.http.ResponseEntity;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import javax.annotation.Nullable;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021-04-18
 */
public interface FileStorageService<T> {

    /**
     * 文件上传
     *
     * @param uploadDTO 上传资源信息
     * @return 文件的唯一标识
     * @throws Exception 上传异常
     */
    FileObjDTO<T> upload(FileUploadDTO<T> uploadDTO) throws Exception;

    /**
     * 申请分片上传
     *
     * @param reqVO 申请文件信息
     * @return 分片上传标识
     */
    Long applyChunk(FileChunkReqVO reqVO);

    /**
     * 上传分片
     *
     * @param file   分片文件，合并时可以为空
     * @param saveVO 保存信息
     * @return 文件信息
     */
    FileObjDTO<T> uploadChunk(@Nullable Resource file, @NotNull FileChunkSaveVO saveVO);

    /**
     * 文件下载
     * <p>
     * 适用于小文件下载
     *
     * @param fileCode  文件唯一标识
     * @param childFlag 子文件标识
     * @throws Exception 调用异常
     */
    HttpEntity<Resource> download(String fileCode, String childFlag) throws Exception;

    /**
     * 文件下载
     * <p>
     * 适用于大文件下载
     *
     * @param fileCode 文件唯一标识
     * @throws Exception 调用异常
     */
    HttpEntity<StreamingResponseBody> downloadStreaming(String fileCode, String childFlag) throws Exception;

    /**
     * 直接打包下载
     *
     * @param param 打包的文件参数
     * @return 文件流
     */
    ResponseEntity<StreamingResponseBody> downloadByPackage(FilePackageParam param) throws Exception;

    /**
     * 申请下载打包文件
     *
     * @param param 打包文件参数
     * @return 申请记录ID
     */
    ApiResult<Long> applyPackage(FilePackageParam param);

    /**
     * 查询打包结果
     *
     * @param applyId 申请记录ID
     * @return 打包结果
     */
    ApiResult<PackageResultVO> packageResult(@NotNull Long applyId);

    /**
     * 下载打包文件
     *
     * @param applyId 申请记录ID
     * @return 文件流
     */
    ResponseEntity<StreamingResponseBody> downloadPackage(@NotNull Long applyId, String packageName);

    /**
     * 文件删除
     *
     * @param fileCode 文件唯一标识
     * @throws Exception 异常
     */
    void delete(String fileCode) throws Exception;

    /**
     * 删除文件
     *
     * @param fileCodes 文件唯一标识
     * @throws Exception 异常
     */
    void delete(List<String> fileCodes) throws Exception;

    /**
     * 判断文件是否存在
     *
     * @param fileCode 文件编码
     * @return 是否存在
     * @throws Exception 异常
     */
    boolean exists(String fileCode) throws Exception;

    /**
     * 判断文件是否都存在
     *
     * @param fileCodes 文件编码
     * @return 是否存在
     * @throws Exception 异常
     */
    boolean exists(List<String> fileCodes) throws Exception;

    /**
     * 获取文件信息
     *
     * @param fileCode 文件唯一标识
     * @return 文件信息
     * @throws Exception 异常
     */
    FileObjDTO<T> getByFileCode(String fileCode) throws Exception;

    /**
     * 查询文件信息
     *
     * @param fileCodes 文件唯一标识
     * @return 文件信息
     * @throws Exception 异常
     */
    List<FileObjDTO<T>> queryByFileCode(List<String> fileCodes) throws Exception;
}
