package com.el.coordinator.boot.fsm.util;

import cn.hutool.core.io.FileUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.el.coordinator.boot.fsm.common.FileTypeEnum;
import com.el.coordinator.boot.fsm.common.UploadFileParam;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateDeserializer;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateSerializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;
import org.springframework.core.io.Resource;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;
import org.springframework.util.unit.DataSize;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.function.Supplier;

/**
 * 文件服务用到的工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2021-04-18
 */
public class FileUploadUtil {
    public static final DateTimeFormatter FORMATTER_MILLSECOND = DateTimeFormatter.ofPattern("yyyyMMddHHmmssSSS");
    public static final DateTimeFormatter FORMATTER_DATETIME = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    public static final SimpleDateFormat SDF_DATE = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    public static final DateTimeFormatter FORMATTER_DATE = DateTimeFormatter.ofPattern("yyyy-MM-dd");

    private FileUploadUtil() {
    }

    /**
     * 上传资源转上传参数
     *
     * @param resource    上传资源
     * @param uploadParam 其它参数
     * @return 上传参数
     * @throws Exception 异常
     */
    public static UploadFileParam<?> resource2UploadFileParam(Resource resource, Map<String, String[]> uploadParam) throws IOException {
        String filename = resource.getFilename();
        UploadFileParam fileParam = UploadFileParam.builder()
                .uploadParam(ObjectUtil.defaultIfNull(uploadParam, Collections.emptyMap()))
                .uploadFile(resource)
                .fileSize(DataSize.ofBytes(resource.contentLength()))
                .originalName(filename)
                .suffix(FileUploadUtil.obtainSuffix(filename))
                .mimeType(FileUploadUtil.obtainMimeType(filename))
                .uploadTime(LocalDateTime.now())
                .build();
        fileParam.setFileType(FileUploadUtil.obtainFileType(fileParam.getMimeType()));

        return fileParam;
    }

    /**
     * 根据文件名获取文件后缀
     *
     * @param filename 文件名称
     * @return 文件后缀
     */
    public static String obtainSuffix(String filename) {
        return StrUtil.isBlank(filename) ? null : FileUtil.extName(filename);
    }

    /**
     * 获取文件的mimeType
     *
     * @param filename 文件名称
     * @return mimeType
     */
    public static String obtainMimeType(String filename) {
        return FileUtil.getMimeType(filename);
    }

    /**
     * 获取文件类型
     *
     * @param mimeType 文件的mime-type
     * @return 文件类型
     */
    public static FileTypeEnum obtainFileType(String mimeType) {
        if (StrUtil.isBlank(mimeType)) {
            return null;
        }

        String type = mimeType.substring(0, mimeType.indexOf("/")).toUpperCase();
        switch (type) {
            case "IMAGE":
                return FileTypeEnum.IMAGE;
            case "AUDIO":
                return FileTypeEnum.AUDIO;
            case "VIDEO":
                return FileTypeEnum.VIDEO;
            default:
                return null;
        }
    }

    /**
     * 重试
     *
     * @param supplier  功能
     * @param predicate 判断是否满足
     * @param timeout   超时时间
     * @param <T>       结果类型
     * @return 结果
     */
    public static <T> T retry(Supplier<T> supplier, Predicate<T> predicate, Duration timeout) {
        long limit = timeout.toMillis();
        long start = System.currentTimeMillis();

        int times = 1;
        T result = null;
        while (true) {
            result = supplier.get();
            if (predicate.test(result)) {
                break;
            }
            if (System.currentTimeMillis() - start > limit) {
                break;
            }

            try {
                TimeUnit.SECONDS.sleep(times);
            } catch (InterruptedException ignore) {
            }
            times++;
        }
        return result;
    }

    /**
     * 创建ObjectMapper
     *
     * @return
     */
    public static ObjectMapper createObjectMapper() {
        return Jackson2ObjectMapperBuilder.json()
                .serializerByType(LocalDateTime.class, new LocalDateTimeSerializer(FORMATTER_DATETIME))
                .deserializerByType(LocalDateTime.class, new LocalDateTimeDeserializer(FORMATTER_DATETIME))
                .serializerByType(LocalDate.class, new LocalDateSerializer(FORMATTER_DATE))
                .deserializerByType(LocalDate.class, new LocalDateDeserializer(FORMATTER_DATE))
                .failOnUnknownProperties(false)
                .failOnEmptyBeans(false)
                .build();
    }
}
