package com.el.coordinator.boot.fsm.util.excel.support;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.map.MapUtil;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.metadata.CellData;
import com.alibaba.excel.metadata.CellExtra;
import com.alibaba.excel.read.listener.ReadListener;
import com.el.coordinator.core.common.exception.BusinessException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateDeserializer;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

/**
 * 数据读取监听器.
 *
 * @author Kaiser（wang shao）
 * @date 2021-03-15
 */
public class DataReadListener implements ReadListener<Map<Integer, Object>> {

    /**
     * 属性行索引，以1为起始
     */
    private Integer attributeRowIndex;
    /**
     * 属性列表
     */
    private List<String> attributeList;

    /**
     * 数据类型
     */
    private Class<?> dataType;

    /**
     * 每次读取的数据数量，如果不大于1，则表示无限制，一次性获取
     */
    private Integer dataSize = 10000;

    /**
     * 获取数据的回调接口
     */
    private Consumer<ImportDataModel> dataConsumer;

    public DataReadListener() {
        init();
    }

    public DataReadListener(Integer attributeRowIndex) {
        this.attributeRowIndex = attributeRowIndex;
        init();
    }

    public DataReadListener(List<String> attributeList) {
        this.attributeList = attributeList;
        init();
    }

    @Override
    public void onException(Exception exception, AnalysisContext context) throws Exception {
        throw exception;
    }

    @Override
    public void invokeHead(Map<Integer, CellData> headMap, AnalysisContext context) {
        // 分析属性列表，取值excel中的配置属性
        // 当attributeRowIndex配置时有效
        if (attributeMap == null && attributeRowIndex != null && attributeRowIndex.intValue() == headIndex
                && MapUtil.isNotEmpty(headMap)) {
            attributeMap = new HashMap<>(headMap.size());
            headMap.forEach((k, v) -> attributeMap.put(k, v.getStringValue()));
        }
        headIndex++;
    }

    @Override
    public void invoke(Map<Integer, Object> data, AnalysisContext context) {
        // 转换属性
        Map<Object, Object> dataConvert = new HashMap<>(data.size());
        data.forEach((key, value) -> {
            if (value == null || "".equals(value.toString().trim())) {
                return;
            }
            Object attr = attributeMap == null ? key : attributeMap.get(key);
            if (attr == null) {
                return;
            }
            dataConvert.put(attr, value);
        });

        // 添加到数据列表
        if (dataType == null || attributeMap == null) {
            dataList.add(dataConvert);
        } else {
            try {
                dataList.add(objectMapper.convertValue(dataConvert, dataType));
            } catch (IllegalArgumentException e) {
                throw new BusinessException("转换数据至bean失败");
            }
        }

        // 判断是否达到回调限制
        if (dataConsumer != null) {
            if (dataSize != null && dataSize.intValue() == dataIndex) {
                dataConsumer.accept(new ImportDataModel(false, dataList));

                // 重置数据列表
                dataList.clear();
                dataIndex = 0;
            }
        }

        dataIndex++;
    }

    @Override
    public void extra(CellExtra extra, AnalysisContext context) {
        // 扩展信息处理
    }

    @Override
    public void doAfterAllAnalysed(AnalysisContext context) {
        if (dataConsumer != null) {
            dataConsumer.accept(new ImportDataModel(true, dataList));
        }
        finish = true;
    }

    @Override
    public boolean hasNext(AnalysisContext context) {
        return true;
    }

    /**
     * 初始化操作
     */
    private void init() {
        // 设置属性
        generateAttributeMap();

        // 初始化数据列表
        dataList = new ArrayList<>(dataSize == null ? 256 : dataSize);
        dataIndex = 1;
    }

    private void generateAttributeMap() {
        if (CollUtil.isNotEmpty(attributeList)) {
            attributeMap = new HashMap<>(attributeList.size());
            var attrIndex = 0;
            for (String attr : attributeList) {
                attributeMap.put(attrIndex++, attr);
            }
        } else {
            if (attributeRowIndex != null && attributeRowIndex < 1) {
                throw new BusinessException("请设置有效的属性参数");
            }
        }
        headIndex = 1;
    }

    public void setDataType(Class<?> dataType) {
        this.dataType = dataType;
    }

    public void setDataSize(Integer dataSize) {
        this.dataSize = dataSize;
    }

    public void setDataConsumer(Consumer<ImportDataModel> dataConsumer) {
        this.dataConsumer = dataConsumer;
    }

    /**
     * 获取数据
     *
     * @return 数据列表
     */
    public List<Object> getDataList() {
        return dataList;
    }

    /**
     * 是否读取完毕
     *
     * @return 是否完毕
     */
    public boolean isFinish() {
        return finish;
    }

    static {
        // 初始化数据转换工具
        objectMapper = Jackson2ObjectMapperBuilder.json()
                .failOnUnknownProperties(false)
                .deserializerByType(LocalDateTime.class, new LocalDateTimeDeserializer(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")))
                .deserializerByType(LocalDate.class, new LocalDateDeserializer(DateTimeFormatter.ofPattern("yyyy-MM-dd")))
                .build();
    }

    /**
     * 当前头部行的索引
     */
    private Integer headIndex;

    private static final ObjectMapper objectMapper;
    /**
     * 属性与列索引的映射，1为起始
     */
    private Map<Integer, String> attributeMap;
    /**
     * 数据行索引
     */
    private Integer dataIndex;
    /**
     * 数据列表
     */
    private List<Object> dataList;
    /**
     * 是否读取完毕
     */
    private boolean finish;
}
