package com.elitescloud.boot.auth.factory.provider;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.auth.factory.common.AuthClientProvider;
import com.elitescloud.boot.auth.factory.common.AuthResult;
import com.elitescloud.boot.auth.factory.common.constant.TokenPositionEnum;
import com.elitescloud.boot.auth.factory.provider.properties.InheritAuthProperty;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import java.util.HashMap;
import java.util.Map;

/**
 * 继承父级.
 *
 * @author Kaiser（wang shao）
 * @date 2025/5/26 周一
 */
public class InheritAuthClientProvider implements AuthClientProvider<InheritAuthProperty> {
    private static final Logger logger = LoggerFactory.getLogger(InheritAuthClientProvider.class);

    @Override
    public String code() {
        return "inherit";
    }

    @Override
    public String name() {
        return "从父级继承";
    }

    @Override
    public Class<InheritAuthProperty> propertyClass() {
        return InheritAuthProperty.class;
    }

    @Override
    public AuthResult authenticate(InheritAuthProperty property) {
        HttpServletRequest request = HttpServletUtil.currentRequest();
        if (request == null) {
            logger.info("当前请求为空，无需继承认证");
            return null;
        }

        AuthResult result = new AuthResult();
        result.setExtra(new HashMap<>(4));

        // 请求头
        if (CollUtil.isNotEmpty(property.getHeaders())) {
            result.setTokenPosition(TokenPositionEnum.HEADER);

            if (property.getHeaders().size() == 1) {
                String header = property.getHeaders().get(0);
                result.setTokenName(header);
                result.setTokenValue(request.getHeader(header));
            } else {
                Map<String, String> headers = new HashMap<>(property.getHeaders().size());
                for (String header : property.getHeaders()) {
                    headers.put(header, request.getHeader(header));
                }
                result.getExtra().put("_headers", headers);
            }
            return result;
        }

        // 查询参数
        if (CollUtil.isNotEmpty(property.getQueryParams())) {
            result.setTokenPosition(TokenPositionEnum.QUERY);
            if (property.getQueryParams().size() == 1) {
                String queryParam = property.getQueryParams().get(0);
                result.setTokenName(queryParam);
                result.setTokenValue(request.getParameter(queryParam));
            } else {
                Map<String, String> queryParams = new HashMap<>(property.getQueryParams().size());
                for (String queryParam : property.getQueryParams()) {
                    queryParams.put(queryParam, request.getParameter(queryParam));
                }
                result.getExtra().put("_query", queryParams);
            }
            return result;
        }

        // Cookie
        if (CollUtil.isNotEmpty(property.getCookies())) {
            result.setTokenPosition(TokenPositionEnum.COOKIE);
            if (property.getCookies().size() == 1) {
                for (Cookie cookie : request.getCookies()) {
                    if (CharSequenceUtil.equals(cookie.getName(), property.getCookies().get(0))) {
                        result.setTokenName(cookie.getName());
                        result.setTokenValue(cookie.getValue());
                        break;
                    }
                }
            } else {
                Map<String, String> cookies = new HashMap<>(property.getCookies().size());
                for (Cookie cookie : request.getCookies()) {
                    if (property.getCookies().contains(cookie.getName())) {
                        cookies.put(cookie.getName(), cookie.getValue());
                    }
                }
                result.getExtra().put("_cookies", cookies);
            }
            return result;
        }

        logger.error("暂不支持的继承类型");
        return null;
    }
}
