package com.elitescloud.boot.auth.provider.config;

import com.elitescloud.boot.auth.client.common.OAuth2ClientConstant;
import com.elitescloud.boot.auth.client.config.AuthorizationProperties;
import com.elitescloud.boot.auth.client.config.CloudtAuthorizationCacheAutoConfiguration;
import com.elitescloud.boot.auth.client.config.security.handler.DelegateAuthenticationCallable;
import com.elitescloud.boot.auth.client.config.support.AuthenticationCache;
import com.elitescloud.boot.auth.client.config.support.AuthenticationCallable;
import com.elitescloud.boot.auth.client.token.AbstractCustomAuthenticationToken;
import com.elitescloud.boot.auth.provider.AuthenticationService;
import com.elitescloud.boot.auth.provider.common.AuthorizationConstant;
import com.elitescloud.boot.auth.provider.config.servlet.ServletOAuth2ServerConfig;
import com.elitescloud.boot.auth.provider.config.servlet.ServletSingleConfig;
import com.elitescloud.boot.auth.provider.config.system.ConfigProperties;
import com.elitescloud.boot.auth.provider.controller.LoginController;
import com.elitescloud.boot.auth.provider.provider.DefaultTokenPropertiesProvider;
import com.elitescloud.boot.auth.provider.provider.user.UserDetailManager;
import com.elitescloud.boot.auth.provider.security.AuthenticationCheckService;
import com.elitescloud.boot.auth.provider.security.TokenPropertiesProvider;
import com.elitescloud.boot.auth.provider.security.generator.token.JwtTokenGenerator;
import com.elitescloud.boot.auth.provider.security.generator.token.TokenGenerator;
import com.elitescloud.boot.auth.provider.security.grant.CredentialCheckable;
import com.elitescloud.boot.auth.provider.security.grant.InternalAuthenticationGranter;
import com.elitescloud.boot.auth.provider.security.handler.CacheUserAuthenticationCallable;
import com.elitescloud.boot.auth.provider.security.handler.LoginLogHandler;
import com.elitescloud.boot.auth.provider.security.handler.LogoutRedirectHandler;
import com.elitescloud.boot.auth.provider.security.impl.DefaultAuthenticationCheckServiceImpl;
import com.elitescloud.boot.auth.provider.security.impl.DefaultAuthenticationService;
import com.elitescloud.boot.auth.provider.security.listener.HttpSessionHolder;
import com.elitescloud.boot.auth.provider.security.listener.SessionEventApplicationListener;
import com.elitescloud.boot.constant.AuthenticationClaim;
import com.elitescloud.boot.log.LogProperties;
import com.elitescloud.boot.log.queue.LogEvent;
import com.elitescloud.boot.util.JwtUtil;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.lmax.disruptor.RingBuffer;
import com.nimbusds.jose.jwk.RSAKey;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.session.SessionProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.servlet.ServletListenerRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.oauth2.jwt.JwtClaimsSet;
import org.springframework.security.oauth2.server.authorization.token.JwtEncodingContext;
import org.springframework.security.oauth2.server.authorization.token.OAuth2TokenCustomizer;
import org.springframework.session.config.SessionRepositoryCustomizer;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.NotNull;
import java.io.IOException;

/**
 * Auth2相关自动化配置.
 *
 * @author Kaiser（wang shao）
 * @date 2021/12/31
 */
@ConditionalOnProperty(prefix = AuthorizationProperties.CONFIG_PREFIX, name = "enabled", havingValue = "true", matchIfMissing = true)
@EnableWebSecurity(debug = false)
@Import({ServletSingleConfig.class, ServletOAuth2ServerConfig.class, SystemConfig.class, SsoConfig.class, Sso2Config.class, LdapConfig.class,
        CloudtCasClientConfig.class, AuthorizationAutoConfiguration.LoginLogConfig.class, AuthorizationAutoConfiguration.HttpSessionConfig.class})
@AutoConfigureAfter(CloudtAuthorizationCacheAutoConfiguration.class)
@Log4j2
public class AuthorizationAutoConfiguration {

    private final AuthorizationProperties authorizationProperties;

    public AuthorizationAutoConfiguration(AuthorizationProperties authorizationProperties) {
        this.authorizationProperties = authorizationProperties;
        Assert.notNull(authorizationProperties.getType(), "未知服务认证方式");

        log.info("服务认证方式：{}", authorizationProperties.getType());
    }

    @Bean
    @ConditionalOnMissingBean
    TokenPropertiesProvider tokenPropertiesProviderDefault() {
        return new DefaultTokenPropertiesProvider(authorizationProperties);
    }

    @Bean
    @ConditionalOnMissingBean
    public TokenGenerator tokenGenerator(RSAKey rsaKey,
                                         OAuth2TokenCustomizer<JwtEncodingContext> jwtCustomizer,
                                         TokenPropertiesProvider tokenPropertiesProvider) {
        var jwtEncoder = JwtUtil.buildJwtEncoder(rsaKey);
        JwtTokenGenerator jwtTokenGenerator = new JwtTokenGenerator(jwtEncoder, tokenPropertiesProvider);
        jwtTokenGenerator.setTokenCustomizer(jwtCustomizer);

        return jwtTokenGenerator;
    }

    /**
     * 内部认证授权
     *
     * @param userDetailManager
     * @param tokenGenerator
     * @return
     */
    @Bean
    @ConditionalOnBean(UserDetailManager.class)
    InternalAuthenticationGranter internalAuthenticationGranter(UserDetailManager userDetailManager,
                                                                TokenGenerator tokenGenerator) {
        InternalAuthenticationGranter granter = new InternalAuthenticationGranter(userDetailManager, tokenGenerator);
        granter.setDelegateAuthenticationCallable(DelegateAuthenticationCallable.getInstance());
        return granter;
    }

    /**
     * 自定义token的claim
     *
     * @return 自定义token的claim
     */
    @Bean
    OAuth2TokenCustomizer<JwtEncodingContext> jwtCustomizer() {
        return context -> {
            Authentication authentication = context.getPrincipal();
            Object principal = authentication.getPrincipal();
            JwtClaimsSet.Builder claims = context.getClaims();

            // 客户端信息
            var client = context.getRegisteredClient();
            if (client != null) {
                claims.claim(AuthenticationClaim.KEY_CLIENT_ID, client.getClientId());

                String tenants = client.getClientSettings().getSetting(OAuth2ClientConstant.SETTING_TENANT);
                if (StringUtils.hasText(tenants)) {
                    claims.claim(AuthenticationClaim.KEY_AUTHED_TENANTS, tenants);
                }
            } else {
                var clientId = (String) HttpServletUtil.currentRequest().getAttribute(AuthorizationConstant.REQUEST_ATTRIBUTE_CLIENT_ID);
                if (StringUtils.hasText(clientId)) {
                    claims.claim(AuthenticationClaim.KEY_CLIENT_ID, clientId);
                }
            }

            // 用户信息
            if (principal instanceof GeneralUserDetails) {
                GeneralUserDetails userDetails = (GeneralUserDetails) principal;

                claims.claim(AuthenticationClaim.KEY_USERNAME, userDetails.getUsername());
                claims.claim(AuthenticationClaim.KEY_USERID, userDetails.getUser().getId());
                if (userDetails.getTenant() != null) {
                    claims.claim(AuthenticationClaim.KEY_TENANT_ID, userDetails.getTenant().getId());
                }
                if (userDetails.getUser().getCasUserId() != null) {
                    claims.claim(AuthenticationClaim.KEY_CAS_USERID, userDetails.getUser().getCasUserId());
                }

                claims.claim(AuthenticationClaim.KEY_PRINCIPAL_TYPE, AuthenticationClaim.VALUE_PRINCIPAL_USER);
            } else {
                // 客户端信息
                claims.claim(AuthenticationClaim.KEY_PRINCIPAL_TYPE, AuthenticationClaim.VALUE_PRINCIPAL_CLIENT);
            }

            // 登录信息
            if (authentication instanceof AbstractCustomAuthenticationToken) {
                var cusAuth = (AbstractCustomAuthenticationToken) authentication;
                claims.claim(AuthenticationClaim.KEY_LOGIN_TYPE, cusAuth.loginType().getType());
                claims.claim(AuthenticationClaim.KEY_TERMINAL, cusAuth.getTerminal() == null ? "" : cusAuth.getTerminal().name());
            }
        };
    }

    /**
     * 登录用户后的缓存处理
     *
     * @param authenticationCache
     * @return
     */
    @Bean
    AuthenticationCallable authenticationCallableCacheUser(AuthenticationCache authenticationCache,
                                                           TokenPropertiesProvider tokenPropertiesProvider) {
        return new CacheUserAuthenticationCallable(authorizationProperties, authenticationCache, tokenPropertiesProvider);
    }

    /**
     * 登录服务相关服务
     *
     * @return
     */
    @Bean
    @ConditionalOnMissingBean
    AuthenticationService authenticationService(@Autowired(required = false) LogoutRedirectHandler logoutRedirectHandler,
                                                @Autowired(required = false) InternalAuthenticationGranter internalAuthenticationGranter) {
        return new DefaultAuthenticationService(DelegateAuthenticationCallable.getInstance(), logoutRedirectHandler, internalAuthenticationGranter);
    }

    /**
     * 登录服务相关接口
     *
     * @param authenticationService
     * @return
     */
    @Bean
    @ConditionalOnBean({AuthenticationService.class})
    LoginController loginSupportController(AuthenticationService authenticationService) {
        return new LoginController(authenticationService);
    }

    /**
     * 用户认证是信息检查
     *
     * @return
     */
    @Bean
    AuthenticationCheckService defaultAuthenticationCheckService(ConfigProperties configProperties) {
        return new DefaultAuthenticationCheckServiceImpl(authorizationProperties, configProperties);
    }

    /**
     * session监听器
     *
     * @return
     */
    @Bean
    HttpSessionHolder httpSessionHolder() {
        return new HttpSessionHolder();
    }
    @Bean
    ServletListenerRegistrationBean<HttpSessionHolder> servletListenerRegistrationBeanHttpSession() {
        ServletListenerRegistrationBean<HttpSessionHolder> listenerRegistrationBean = new ServletListenerRegistrationBean<>();
        listenerRegistrationBean.setListener(httpSessionHolder());
        listenerRegistrationBean.setEnabled(true);
        return listenerRegistrationBean;
    }

    /**
     * 密码认证校验
     *
     * @return
     */
    @Bean
    @ConditionalOnMissingBean
    CredentialCheckable credentialCheckable() {
        return new CredentialCheckable() {
            @Override
            public <T extends AbstractCustomAuthenticationToken<T>> boolean needCheck(T authenticationToken, GeneralUserDetails userDetails) {
                return true;
            }
        };
    }

    /**
     * 登录日志相关配置
     */
    @ConditionalOnProperty(prefix = LogProperties.CONFIG_PREFIX + ".login-log", name = "enabled", havingValue = "true")
    @EnableConfigurationProperties(LogProperties.class)
    static class LoginLogConfig {
        private final LogProperties logProperties;
        private final RingBuffer<LogEvent> ringBuffer;

        public LoginLogConfig(LogProperties logProperties, RingBuffer<LogEvent> ringBuffer) {
            this.logProperties = logProperties;
            this.ringBuffer = ringBuffer;
            log.info("启用登录日志纪录");
        }

        @Bean
        public AuthenticationCallable authenticationCallbackLoginLog(LoginLogHandler loginLogHandler) {
            return new AuthenticationCallable() {
                @Override
                public void onLogin(HttpServletRequest request, HttpServletResponse response, String token, Authentication authentication) throws IOException, ServletException {
                    loginLogHandler.loginLog(request, authentication, null);
                }

                @Override
                public void onLoginFailure(HttpServletRequest request, HttpServletResponse response, @NotNull Authentication authentication, @NotNull AuthenticationException exception) {
                    loginLogHandler.loginLog(request, null, exception);
                }
            };
        }

        @Bean
        public LoginLogHandler loginLogHandler() {
            return new LoginLogHandler(ringBuffer, logProperties);
        }
    }

    @ConditionalOnClass(SessionRepositoryCustomizer.class)
    static class HttpSessionConfig {
        private final SessionProperties sessionProperties;

        public HttpSessionConfig(SessionProperties sessionProperties) {
            this.sessionProperties = sessionProperties;
        }

        @Bean
        SessionEventApplicationListener sessionEventApplicationListener() {
            return new SessionEventApplicationListener();
        }
    }
}
