package com.elitescloud.boot.auth.provider.config;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.auth.cas.AuthorizeCacheable;
import com.elitescloud.boot.auth.cas.model.AuthorizeDTO;
import com.elitescloud.boot.auth.cas.provider.OAuth2ClientTemplate;
import com.elitescloud.boot.auth.cas.provider.UserTransferHelper;
import com.elitescloud.boot.auth.client.common.AuthorizationException;
import com.elitescloud.boot.auth.client.config.AuthorizationProperties;
import com.elitescloud.boot.auth.client.config.support.AuthenticationCallable;
import com.elitescloud.boot.auth.client.token.AbstractCustomAuthenticationToken;
import com.elitescloud.boot.auth.client.tool.RedisHelper;
import com.elitescloud.boot.auth.config.AuthorizationSdkProperties;
import com.elitescloud.boot.auth.provider.cas.controller.CasSupportController;
import com.elitescloud.boot.auth.provider.cas.support.CasLoginSupportProvider;
import com.elitescloud.boot.auth.provider.cas.support.CasTokenPropertiesProvider;
import com.elitescloud.boot.auth.provider.common.AuthorizationConstant;
import com.elitescloud.boot.auth.provider.security.TokenPropertiesProvider;
import com.elitescloud.boot.auth.provider.security.grant.CredentialCheckable;
import com.elitescloud.boot.auth.provider.security.grant.InternalAuthenticationGranter;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.security.authentication.AccountStatusException;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.userdetails.UsernameNotFoundException;

import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 统一身份认证客户端相关配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/12
 */
@Import({CloudtCasClientConfig.CasClientSupportConfig.class})
class CloudtCasClientConfig {
    private final AuthorizationSdkProperties sdkProperties;

    public CloudtCasClientConfig(AuthorizationSdkProperties sdkProperties) {
        this.sdkProperties = sdkProperties;
    }

    @Bean
    public CasLoginSupportProvider casLoginSupportProvider(AuthorizationProperties authorizationProperties,
                                                           @Autowired(required = false) OAuth2ClientTemplate oAuth2ClientTemplate,
                                                           @Autowired(required = false) InternalAuthenticationGranter internalAuthenticationGranter) {
        return new CasLoginSupportProvider(authorizationProperties, sdkProperties, oAuth2ClientTemplate, internalAuthenticationGranter);
    }

    @Bean
    public CasSupportController oAuth2LoginSupportController(CasLoginSupportProvider supportProvider) {
        return new CasSupportController(supportProvider);
    }

    @Slf4j
    @ConditionalOnProperty(prefix = AuthorizationSdkProperties.CONFIG_PREFIX + ".cas-client", name = "enabled", havingValue = "true")
    static class CasClientSupportConfig {
        private final AuthorizationSdkProperties sdkProperties;

        public CasClientSupportConfig(AuthorizationSdkProperties sdkProperties) {
            this.sdkProperties = sdkProperties;
        }

        @Bean
        public AuthenticationCallable authenticationCallableCasClient(AuthorizationSdkProperties sdkProperties) {
            String authUrl = sdkProperties.getAuthServer();
            return new AuthenticationCallable() {
                @Override
                public void onLoginFailure(HttpServletRequest request, HttpServletResponse response, @Nullable Authentication authentication, @NotNull AuthenticationException exception) {
                    log.info("用户认证异常：", exception);
                    if (exception instanceof UsernameNotFoundException || exception instanceof AccountStatusException) {
                        var help = UserTransferHelper.getInstance(authUrl);
                        // 出现异常，则禁用账号
                        if (authentication instanceof InternalAuthenticationGranter.InternalAuthenticationToken) {
                            InternalAuthenticationGranter.InternalAuthenticationToken authenticationToken = (InternalAuthenticationGranter.InternalAuthenticationToken) authentication;
                            Long userId = null;
                            if (authenticationToken.getIdType() == InternalAuthenticationGranter.IdType.USER_ID) {
                                userId = Long.parseLong(authenticationToken.getId());
                            } else if (authenticationToken.getIdType() == InternalAuthenticationGranter.IdType.USERNAME) {
                                var user = help.getUserByUsername(authenticationToken.getId()).getData();
                                if (user == null) {
                                    log.error("自动禁用账号失败，账号{}不存在", authenticationToken.getId());
                                    return;
                                }
                                userId = user.getId();
                            } else if (authenticationToken.getIdType() == InternalAuthenticationGranter.IdType.MOBILE) {
                                var result = help.getUserIdByMobile(List.of(authenticationToken.getId()));
                                if (result.getData() == null || result.getData().isEmpty()) {
                                    log.error("自动禁用账号失败，{}", result.getMsg());
                                    return;
                                }
                                userId = result.getData().get(authenticationToken.getId());
                            } else if (authenticationToken.getIdType() == InternalAuthenticationGranter.IdType.EMAIL) {
                                var result = help.getUserIdByEmail(List.of(authenticationToken.getId()));
                                if (result.getData() == null || result.getData().isEmpty()) {
                                    log.error("自动禁用账号失败，{}", result.getMsg());
                                    return;
                                }
                                userId = result.getData().get(authenticationToken.getId());
                            }

                            if (userId != null) {
                                var result = help.updateEnabled(userId, false);
                                if (Boolean.FALSE.equals(result.getSuccess())) {
                                    log.error("自动禁用账号失败：" + result.getMsg());
                                }
                                return;
                            }
                            log.error("暂不支持自动禁用账号：" + authenticationToken.getIdType());
                        }
                    }
                }
            };
        }

        @Bean
        public TokenPropertiesProvider casTokenPropertiesProvider(AuthorizationProperties authorizationProperties) {
            return new CasTokenPropertiesProvider(authorizationProperties, sdkProperties);
        }

        @Bean
        @ConditionalOnBean(RedisHelper.class)
        public AuthorizeCacheable authorizeCacheableRedis(RedisHelper redisHelper) {
            return new AuthorizeCacheable() {
                private static final String KEY_PREFIX = "cas:authorize:";

                @Override
                public void setCache(String reqId, AuthorizeDTO authorizeDTO) {
                    try {
                        redisHelper.execute(redisUtils -> redisUtils.set(KEY_PREFIX + reqId, authorizeDTO, 7, TimeUnit.DAYS));
                    } catch (Exception e) {
                        throw new IllegalStateException("登录异常", e);
                    }
                }

                @Override
                public AuthorizeDTO get(String reqId) {
                    try {
                        return (AuthorizeDTO) redisHelper.execute(redisUtils -> redisUtils.get(KEY_PREFIX + reqId));
                    } catch (Exception e) {
                        throw new IllegalStateException("登录异常", e);
                    }
                }
            };
        }

        @Bean
        public CredentialCheckable credentialCheckableCasClient() {
            return new CredentialCheckable() {
                private final UserTransferHelper userTransferHelper = UserTransferHelper.getInstance(sdkProperties.getAuthServer());

                @Override
                public <T extends AbstractCustomAuthenticationToken<T>> boolean needCheck(T authenticationToken, GeneralUserDetails userDetails) {
                    String pwd = (String) authenticationToken.getCredentials();
                    if (CharSequenceUtil.isBlank(pwd)) {
                        throw new BadCredentialsException("账号或密码不正确");
                    }

                    var casUserId = userDetails.getUser().getCasUserId();

                    // rpc校验
                    var validateUserResult = userTransferHelper.validateLogin(casUserId, pwd);
                    if (!Boolean.TRUE.equals(validateUserResult.getSuccess())) {
                        throw new AuthorizationException(validateUserResult.getMsg());
                    }
                    var request = HttpServletUtil.currentRequest();
                    Assert.notNull(request, "请求为空");
                    request.setAttribute(AuthorizationConstant.REQUEST_ATTRIBUTE_AUTH_USER_DETAIL_EXTEND, validateUserResult.getData());

                    if (casUserId == null) {
                        log.info("未向CAS同步的账号，走本地密码校验");
                        return true;
                    }

                    var user = userDetails.getUser();
                    if (Boolean.TRUE.equals(user.getEnabled())) {
                        user.setEnabled(Boolean.TRUE.equals(validateUserResult.getData().getEnabled()));
                    }
                    user.setLocked(Boolean.TRUE.equals(validateUserResult.getData().getLocked()));
                    user.setPwdExpiredTime(validateUserResult.getData().getPwdExpiredTime());
                    user.setNeedReset(validateUserResult.getData().getNeedReset());

                    // 不需要再本地校验密码
                    return false;
                }
            };
        }
    }
}
