package com.elitescloud.boot.auth.provider.config.system;

import cn.hutool.captcha.AbstractCaptcha;
import cn.hutool.captcha.CaptchaUtil;
import com.elitescloud.boot.auth.provider.common.LoginAccountType;
import com.elitescloud.boot.auth.provider.config.system.captcha.CaptchaCodeEnum;
import com.wf.captcha.*;
import com.wf.captcha.base.Captcha;
import lombok.Data;
import org.springframework.boot.context.properties.NestedConfigurationProperty;

import java.io.Serializable;
import java.util.Objects;
import java.util.Set;

/**
 * @author Michael Li
 * @date AC 2021-04-12:14:13
 */
public class LoginProperties implements Serializable {
    private static final long serialVersionUID = 3572914221247822407L;

    /**
     * 验证码
     */
    @NestedConfigurationProperty
    private CaptchaInfo captchaInfo = new CaptchaInfo();

    /**
     * 是否开启终端限制
     */
    private Boolean terminalLimit = false;

    /**
     * 是否允许自动注册用户
     */
    private Boolean autoRegister = false;

    /**
     * 登录账号类型
     */
    private Set<LoginAccountType> loginAccountTypes = Set.of(LoginAccountType.USERNAME, LoginAccountType.MOBILE);

    public LoginProperties() {
    }

    public AbstractCaptcha getCaptchaHu() {
        if (Objects.isNull(captchaInfo)) {
            return null;
        }

        if (captchaInfo.getCodeType() == CaptchaCodeEnum.line || captchaInfo.getCodeType() == CaptchaCodeEnum.circle || captchaInfo.getCodeType() == CaptchaCodeEnum.shear) {
            switch (captchaInfo.getCodeType()) {
                case line:
                    return CaptchaUtil.createLineCaptcha(captchaInfo.getWidth(), captchaInfo.getHeight());
                case circle:
                    return CaptchaUtil.createCircleCaptcha(captchaInfo.getWidth(), captchaInfo.getHeight());
                default:
                    return CaptchaUtil.createShearCaptcha(captchaInfo.getWidth(), captchaInfo.getHeight());
            }
        }

        return null;
    }

    /**
     * 获取验证码生产类
     *
     * @return /
     */
    public Captcha getCaptcha() {
        if (Objects.isNull(captchaInfo)) {
            captchaInfo = new CaptchaInfo();
            if (Objects.isNull(captchaInfo.getCodeType())) {
                captchaInfo.setCodeType(CaptchaCodeEnum.arithmetic);
            }
        }
        return switchCaptcha(captchaInfo);
    }

    /**
     * 依据配置信息生产验证码
     *
     * @param captchaInfo 验证码配置信息
     * @return /
     */
    private Captcha switchCaptcha(CaptchaInfo captchaInfo) {
        Captcha captcha;
        synchronized (this) {
            switch (captchaInfo.getCodeType()) {
                case arithmetic:
                    // 算术类型 https://gitee.com/whvse/EasyCaptcha
                    captcha = new ArithmeticCaptcha(captchaInfo.getWidth(), captchaInfo.getHeight());
                    // 几位数运算，默认是两位
                    captcha.setLen(captchaInfo.getLength());
                    break;
                case chinese:
                    captcha = new ChineseCaptcha(captchaInfo.getWidth(), captchaInfo.getHeight());
                    captcha.setLen(captchaInfo.getLength());
                    break;
                case chinese_gif:
                    captcha = new ChineseGifCaptcha(captchaInfo.getWidth(), captchaInfo.getHeight());
                    captcha.setLen(captchaInfo.getLength());
                    break;
                case gif:
                    captcha = new GifCaptcha(captchaInfo.getWidth(), captchaInfo.getHeight());
                    captcha.setLen(captchaInfo.getLength());
                    break;
                case spec:
                    captcha = new SpecCaptcha(captchaInfo.getWidth(), captchaInfo.getHeight());
                    captcha.setLen(captchaInfo.getLength());
                    break;
                default:
                    throw new RuntimeException("验证码配置信息错误！！！正确配置查看 " + CaptchaCodeEnum.class.getName());
            }
        }
        return captcha;
    }

    public CaptchaInfo getCaptchaInfo() {
        return captchaInfo;
    }

    public void setCaptchaInfo(CaptchaInfo captchaInfo) {
        this.captchaInfo = captchaInfo;
    }

    public Boolean getTerminalLimit() {
        return terminalLimit;
    }

    public void setTerminalLimit(Boolean terminalLimit) {
        this.terminalLimit = terminalLimit;
    }

    public Boolean getAutoRegister() {
        return autoRegister;
    }

    public void setAutoRegister(Boolean autoRegister) {
        this.autoRegister = autoRegister;
    }

    public Set<LoginAccountType> getLoginAccountTypes() {
        return loginAccountTypes;
    }

    public void setLoginAccountTypes(Set<LoginAccountType> loginAccountTypes) {
        this.loginAccountTypes = loginAccountTypes;
    }
}
