package com.elitescloud.boot.auth.provider.provider;

import com.elitescloud.boot.auth.provider.config.system.CaptchaEnum;
import com.elitescloud.boot.auth.provider.config.system.CaptchaInfo;
import com.elitescloud.boot.auth.provider.config.system.ConfigProperties;
import com.elitescloud.boot.auth.provider.config.system.LoginProperties;
import com.elitescloud.boot.auth.provider.resp.CaptchaRespVO;
import com.elitescloud.boot.common.param.SysSendVerifyCodeVO;
import com.elitescloud.boot.core.support.verifycode.common.VerifyCodeManager;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.boot.exception.BusinessException;
import lombok.extern.log4j.Log4j2;
import lombok.val;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.StringUtils;

import java.time.Duration;

/**
 * 通用登录相关扩展支撑.
 *
 * @author Kaiser（wang shao）
 * @date 2022/01/08
 */
@Log4j2
public class LoginSupportProvider {

    private final ConfigProperties configProperties;

    private CaptchaProvider captchaManager;
    private VerifyCodeManager verifyCodeManager;

    public LoginSupportProvider(ConfigProperties configProperties) {
        this.configProperties = configProperties;
    }

    /**
     * 生成验证码
     *
     * @return 验证码
     */
    public ApiResult<CaptchaRespVO> generateCaptcha() {
        LoginProperties loginProperties = configProperties.getLogin();
        CaptchaInfo captchaInfo = loginProperties.getCaptchaInfo();
        CaptchaRespVO respVO = null;

        // 验证码的有效期
        var timeToLive = Duration.ofMinutes(captchaInfo.getExpiration());

        val captchaHu = loginProperties.getCaptchaHu();
        if (captchaHu != null) {
            respVO = getCaptchaManager().generateCaptchaHu(CaptchaEnum.LOGIN, captchaHu, timeToLive);
        } else {
            respVO = getCaptchaManager().generateCaptcha(CaptchaEnum.LOGIN, loginProperties.getCaptcha(), timeToLive);
        }

        log.info("验证码：{}", respVO.getCaptext());
        return ApiResult.ok(respVO);
    }

    /**
     * 校验验证码
     *
     * @param captchaKey  验证码key
     * @param captchaText 验证码内容
     * @return 校验结果
     */
    public ApiResult<Boolean> verifyCaptcha(String captchaKey, String captchaText) {
        if (!StringUtils.hasText(captchaKey)) {
            return ApiResult.fail("未设置验证码");
        }

        if (!StringUtils.hasText(captchaText)) {
            return ApiResult.fail("验证码为空");
        }

        boolean verify = getCaptchaManager().verifyCaptcha(CaptchaEnum.LOGIN, captchaKey, captchaText);
        return ApiResult.ok(verify);
    }

    /**
     * 发送验证码
     *
     * @param verifyCodeVO 验证码参数
     * @return 发送结果
     */
    public ApiResult<String> sendVerifyCode(SysSendVerifyCodeVO verifyCodeVO) {
        String code = verifyCodeManager.send(VerifyCodeManager.BUSINESS_TYPE_AUTH_LOGIN, verifyCodeVO);

        String msg = code == null ? "发送成功" : "模拟发送验证码，验证码是" + code;
        return ApiResult.ok(msg);
    }

    public CaptchaProvider getCaptchaManager() {
        if (captchaManager == null) {
            throw new BusinessException("验证码服务未初始化");
        }
        return captchaManager;
    }

    @Autowired(required = false)
    public void setCaptchaManager(CaptchaProvider captchaManager) {
        this.captchaManager = captchaManager;
    }

    @Autowired
    public void setVerifyCodeManager(VerifyCodeManager verifyCodeManager) {
        this.verifyCodeManager = verifyCodeManager;
    }
}
