package com.elitescloud.boot.auth.provider.provider.alipay;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import com.alipay.api.response.AlipaySystemOauthTokenResponse;
import com.elitescloud.boot.auth.client.tool.RedisHelper;
import com.elitescloud.boot.auth.provider.common.AlipayAppProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.JSONUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.validation.constraints.NotBlank;
import java.util.Map;

/**
 * 支付宝工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2025/2/5
 */
public class AlipayTemplate {
    private static final Logger logger = LoggerFactory.getLogger(AlipayTemplate.class);

    private final AlipayAppProvider appProvider;
    private final RedisHelper redisHelper;

    public AlipayTemplate(AlipayAppProvider appProvider, RedisHelper redisHelper) {
        this.appProvider = appProvider;
        this.redisHelper = redisHelper;
    }

    /**
     * 获取授权令牌
     * <p>
     * 包含access_token和openid
     *
     * @param appId 应用ID
     * @param code 支付宝授权码
     * @return 授权令牌
     */
    public AlipaySystemOauthTokenResponse oauthToken(@NotBlank String appId, @NotBlank String code) {
        Assert.notBlank(appId, "应用ID为空");
        Assert.notBlank(code, "支付宝授权码为空");

        logger.info("获取授权令牌：{}，{}", appId, code);

        // 获取支付宝配置
        var app = appProvider.getApp(appId);
        if (app == null) {
            throw new BusinessException("支付宝应用不存在或未配置");
        }

        try {
            return AlipayTool.systemOAuthToken(app, code);
        } catch (Exception e) {
            throw new BusinessException("支付宝授权失败", e);
        }
    }

    /**
     * 解析手机号
     *
     * @param appId         应用ID
     * @param encryptedResp 支付宝响应内容
     * @return 手机号
     * @see <a href='https://opendocs.alipay.com/mini/api/getphonenumber?pathHash=a67c2790'>官方文档</a>
     */
    public String getPhoneNumber(@NotBlank String appId, @NotBlank String encryptedResp) throws Exception {
        Assert.notBlank(appId, "应用ID为空");
        Assert.notBlank(encryptedResp, "支付宝响应内容为空");

        logger.info("获取手机号：{}，{}", appId, encryptedResp);

        // 获取支付宝配置
        var app = appProvider.getApp(appId);
        if (app == null) {
            throw new BusinessException("支付宝应用不存在或未配置");
        }

        // 获取响应信息
        Map<String, Object> resp = JSONUtil.json2Obj(encryptedResp, true, () -> "解析支付宝响应内容异常");
        Object response = resp.get("response");
        if (!(response instanceof String)) {
            Map<String, Object> responseMap = (Map<String, Object>) response;
            throw new BusinessException("系统异常，支付宝响应：" + responseMap.get("code") + ", " + responseMap.get("subCode") + ", " + responseMap.get("subMsg"));
        }
        String sign = (String) resp.get("sign");

        // 验签
        logger.info("验签： {}, {}, {}", response, sign, app.getPublicKeyAlipay());
        if (CharSequenceUtil.isNotBlank(sign)) {
            String content = (String) response;
            String signContent = content.startsWith("{") ? content : "\"" + content + "\"";
            if (!AlipayTool.RSA2.verifySign(signContent, sign, app.getPublicKeyAlipay())) {
                throw new BusinessException("系统异常，验证响应内容失败");
            }
        }

        // 解密响应信息
        String decryptedResp = AlipayTool.AES.decrypt(app.getAesKey(), (String) response);
        logger.info("解密响应信息：{}", decryptedResp);
        Map<String, Object> data = JSONUtil.json2Obj(decryptedResp, true, () -> "解析支付宝响应内容异常");
        if (data.containsKey("subMsg") && !data.containsKey("mobile")) {
            throw new BusinessException("获取用户手机号失败，" + data.containsKey("subMsg"));
        }

        return (String) data.get("mobile");
    }
}
