package com.elitescloud.boot.auth.provider.security.grant.wecom;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.auth.client.common.AuthorizationException;
import com.elitescloud.boot.auth.provider.common.LoginAccountType;
import com.elitescloud.boot.auth.provider.provider.wecom.WecomTemplate;
import com.elitescloud.boot.auth.provider.security.grant.AbstractCustomAuthenticationProvider;
import com.elitescloud.boot.auth.provider.security.jackson.mixin.grant.MixinWecomAuthenticationToken;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.AuthenticationException;
import org.springframework.util.StringUtils;

/**
 * 企微的身份认证.
 *
 * @author Kaiser（wang shao）
 * @date 2022/01/01
 */
@Log4j2
public class WecomAuthenticationProvider extends AbstractCustomAuthenticationProvider<WecomAuthenticationToken> {

    @Autowired
    private WecomTemplate wecomTemplate;

    @Override
    protected GeneralUserDetails retrieveUser(WecomAuthenticationToken authentication) throws AuthenticationException {
        String corpId = (String) authentication.getPrincipal();
        if (!StringUtils.hasText(corpId)) {
            throw new AuthorizationException("appId为空");
        }
        LoginAccountType accountType = authentication.getAccountType();
        if (accountType == null) {
            throw new AuthorizationException("账号类型为空");
        }
        String agentId = authentication.getAgentId();
        if (!StringUtils.hasText(agentId)) {
            throw new AuthorizationException("应用ID为空");
        }
        String code = (String) authentication.getCredentials();
        if (!StringUtils.hasText(code)) {
            throw new AuthorizationException("授权码code为空");
        }

        // 调用企微认证，获取基本信息
        var code2UserInfoResult = wecomTemplate.code2UserInfo(corpId, agentId, code);
        if (!code2UserInfoResult.isSuccess()) {
            throw new AuthorizationException(String.format("调用企业微信服务器认证失败[%s,%s]", code2UserInfoResult.getErrcode(), code2UserInfoResult.getErrmsg()));
        }

        if (accountType == LoginAccountType.ID || accountType == LoginAccountType.USERNAME) {
            var userId = code2UserInfoResult.getUserid();
            if (CharSequenceUtil.isBlank(userId)) {
                throw new AuthorizationException("企业微信认证失败");
            }
            return accountType == LoginAccountType.ID ? userDetailManager.loadUserById(userId) : userDetailManager.loadUserByUsername(userId);
        }

        // 手机号、邮箱的话需要详细信息
        if (!StringUtils.hasText(code2UserInfoResult.getUser_ticket())) {
            throw new AuthorizationException("用户票据为空，请确认已授权手机号或邮箱");
        }
        var userInfoDetailResult = wecomTemplate.getUserByTicket(corpId, agentId, code2UserInfoResult.getUser_ticket());
        if (!userInfoDetailResult.isSuccess()) {
            throw new AuthorizationException(String.format("调用企业微信服务器认证失败[%s,%s]", code2UserInfoResult.getErrcode(), code2UserInfoResult.getErrmsg()));
        }

        // 手机号
        if (accountType == LoginAccountType.MOBILE) {
            if (!StringUtils.hasText(userInfoDetailResult.getMobile())) {
                throw new AuthorizationException("获取企微中的手机号失败");
            }
            return userDetailManager.loadUserByMobile(userInfoDetailResult.getMobile());
        }
        // 邮箱
        if (accountType == LoginAccountType.EMAIL) {
            String email = CharSequenceUtil.blankToDefault(userInfoDetailResult.getBiz_mail(), userInfoDetailResult.getEmail());
            if (!StringUtils.hasText(email)) {
                throw new AuthorizationException("获取企微中的邮箱失败");
            }
            return userDetailManager.loadUserByEmail(email);
        }

        throw new AuthorizationException("暂不支持的账号类型" + accountType);
    }

    @Override
    public Class<WecomAuthenticationToken> getAuthenticationTokenType() {
        return WecomAuthenticationToken.class;
    }

    @Override
    public Class<?> getMixinAuthenticationTokenType() {
        return MixinWecomAuthenticationToken.class;
    }
}
