package com.elitescloud.boot.auth.provider.security.handler;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.extra.servlet.ServletUtil;
import com.elitescloud.boot.auth.client.token.AbstractCustomAuthenticationToken;
import com.elitescloud.boot.auth.model.OAuthToken;
import com.elitescloud.boot.auth.provider.common.AuthorizationConstant;
import com.elitescloud.boot.auth.provider.common.LoginMethodEnum;
import com.elitescloud.boot.auth.provider.common.LoginParameterNames;
import com.elitescloud.boot.log.LogProperties;
import com.elitescloud.boot.log.model.bo.LoginLogBO;
import com.elitescloud.boot.log.queue.LogEvent;
import com.elitescloud.boot.threadpool.common.ThreadPoolHolder;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.context.util.DesensitizeUtil;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.lmax.disruptor.RingBuffer;
import lombok.extern.log4j.Log4j2;
import org.springframework.http.HttpHeaders;
import org.springframework.security.core.Authentication;
import org.springframework.util.CollectionUtils;
import org.springframework.util.MultiValueMap;

import javax.servlet.http.HttpServletRequest;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;

/**
 * 登录日志处理.
 *
 * @author Kaiser（wang shao）
 * @date 2022/8/23
 */
@Log4j2
public class LoginLogHandler {

    private final RingBuffer<LogEvent> ringBuffer;
    private final LogProperties logProperties;
    private final Executor executor;

    public LoginLogHandler(RingBuffer<LogEvent> ringBuffer, LogProperties logProperties) {
        this.ringBuffer = ringBuffer;
        this.logProperties = logProperties;
        this.executor = createExecutor();
    }

    public void loginLog(HttpServletRequest request, Authentication authentication, Throwable throwable) {
        createLoginLog(request, authentication, throwable)
                .thenAccept(this::addToQueue)
                .exceptionally((e) -> {
                    log.error("记录登录日志异常：", e);
                    return null;
                });
    }

    private CompletableFuture<LoginLogBO> createLoginLog(HttpServletRequest request, Authentication authentication, Throwable throwable) {
        LocalDateTime now = LocalDateTime.now();
        return CompletableFuture.supplyAsync(() -> {
            LoginLogBO logBO = new LoginLogBO();
            LocalDateTime loginTime = (LocalDateTime) request.getAttribute(AuthorizationConstant.REQUEST_ATTRIBUTE_LOGIN_START_TIME);
            ApiResult<OAuthToken> tokenResult = (ApiResult<OAuthToken>) request.getAttribute(AuthorizationConstant.REQUEST_ATTRIBUTE_LOGIN_RESULT);
            logBO.setRequestTime(ObjectUtil.defaultIfNull(loginTime, now));
            logBO.setResponseTime(now);
            if (tokenResult != null && tokenResult.getData() != null) {
                logBO.setToken(tokenResult.getData().getAccessToken());
            }
            logBO.setUserAgent(request.getHeader(HttpHeaders.USER_AGENT));
            logBO.setMethod(request.getMethod());
            logBO.setReqContentType(request.getContentType());
            logBO.setUri(request.getRequestURI());
            logBO.setOperation("用户登录");
            logBO.setReqIp(ServletUtil.getClientIP(request));
            logBO.setReqOuterIp(request.getRemoteAddr());
            logBO.setQueryParams(request.getQueryString());

            var requestBody = HttpServletUtil.getFormParameters(request);
            // 脱密处理
            desensitizeRequestBody(requestBody);
            logBO.setRequestBody(List.of(normalizeRequestBody(requestBody)));

            if (tokenResult != null) {
                logBO.setResultCode(tokenResult.getCode());
                logBO.setMsg(tokenResult.getMsg());
                logBO.setResult(tokenResult);
            }
            logBO.setThrowable(throwable);

            if (authentication != null) {
                if (authentication instanceof AbstractCustomAuthenticationToken) {
                    logBO.setLoginMethod(LoginMethodEnum.INNER.name());
                }
            }
            logBO.setLoginType(requestBody.getFirst(LoginParameterNames.LOGIN_TYPE));
            logBO.setTerminal(requestBody.getFirst(LoginParameterNames.TERMINAL));
            if (logProperties.getLoginLog().getUserDetail()) {
                if (authentication != null && authentication.getPrincipal() instanceof GeneralUserDetails) {
                    logBO.setUserDetailObj(authentication.getPrincipal());
                }
            }

            return logBO;
        }, executor);
    }

    private void desensitizeRequestBody(MultiValueMap<String, String> body) {
        if (body.isEmpty()) {
            return;
        }

        // 登录密码
        List<String> passwords = body.get(LoginParameterNames.PASSWORD);
        if (!CollectionUtils.isEmpty(passwords)) {
            passwords = passwords.stream().map(DesensitizeUtil::password).collect(Collectors.toList());
            body.put(LoginParameterNames.PASSWORD, passwords);
        }

        // 手机号
        List<String> mobiles = body.get(LoginParameterNames.MOBILE);
        if (!CollectionUtils.isEmpty(mobiles)) {
            mobiles = mobiles.stream().map(DesensitizeUtil::mobile).collect(Collectors.toList());
            body.put(LoginParameterNames.MOBILE, mobiles);
        }
    }

    private Map<String, Object> normalizeRequestBody(MultiValueMap<String, String> body) {
        Map<String, Object> result = new HashMap<>(body.size());
        for (Map.Entry<String, List<String>> entry : body.entrySet()) {
            if (CollectionUtils.isEmpty(entry.getValue())) {
                result.put(entry.getKey(), null);
                continue;
            }

            if (entry.getValue().size() == 1) {
                result.put(entry.getKey(), entry.getValue().get(0));
                continue;
            }

            result.put(entry.getKey(), entry.getValue());
        }
        return result;
    }

    private void addToQueue(LoginLogBO logBO) {
        var msgSequence = ringBuffer.next();
        try {
            ringBuffer.get(msgSequence)
                    .setLog(logBO);
        } catch (Exception e) {
            log.error("添加登录日志到队列时异常：", e);
        } finally {
            ringBuffer.publish(msgSequence);
        }
    }

    private Executor createExecutor() {
        var threadPool = logProperties.getThreadPool();
        return ThreadPoolHolder.createThreadPool(threadPool.getThreadNamePrefix(), threadPool.getCoreSize(), threadPool.getMaxSize());
    }
}
