package com.elitescloud.boot.auth.provider.security.handler.sso;

import com.elitescloud.boot.auth.client.config.support.AuthenticationCallable;
import com.elitescloud.boot.auth.config.AuthorizationSdkProperties;
import com.elitescloud.boot.auth.provider.security.generator.ticket.TicketGenerator;
import com.elitescloud.boot.auth.sso.TicketResolver;
import com.elitescloud.boot.auth.sso.common.SdkSsoConstants;
import com.elitescloud.boot.auth.util.AuthorizationServerHelper;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.log4j.Log4j2;
import org.springframework.security.core.Authentication;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.concurrent.CompletableFuture;

/**
 * cookie关于登录相关处理.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/15
 */
@Log4j2
public class SsoTicketAuthenticationCallable implements AuthenticationCallable {

    private final AuthorizationSdkProperties sdkProperties;
    private final TicketGenerator ticketGenerator;
    private final TicketResolver ticketResolver;

    private final AuthorizationServerHelper authorizationServerHelper = AuthorizationServerHelper.getInstance();

    public SsoTicketAuthenticationCallable(AuthorizationSdkProperties sdkProperties, TicketGenerator ticketGenerator, TicketResolver ticketResolver) {
        this.sdkProperties = sdkProperties;
        this.ticketGenerator = ticketGenerator;
        this.ticketResolver = ticketResolver;
    }

    @Override
    public void onLogin(HttpServletRequest request, HttpServletResponse response, String token, Authentication authentication) throws IOException, ServletException {
        if (response == null) {
            return;
        }
        if (authentication.getPrincipal() instanceof GeneralUserDetails) {
            writeTicket(request, response, (GeneralUserDetails) authentication.getPrincipal());
        }
    }

    @Override
    public void onLogout(HttpServletRequest request, HttpServletResponse response, String token, Object principal) {
        String ticket = ticketResolver.obtain(request);
        if (!StringUtils.hasText(ticket)) {
            log.info("没有单点登录ticket");
            return;
        }

        // 清理本地ticket
        ticketResolver.clear(request, response);
        // 调用sso server进行全局注销
        CompletableFuture.runAsync(() -> {
            // 先删除ticket
            try {
                ticketGenerator.remove(ticket);
            } catch (Exception e) {
                log.error("ticket删除异常", e);
            }

            // 调用各业务系统注销
            if (CollectionUtils.isEmpty(sdkProperties.getSso().getClientAddress())) {
                log.warn("未配置单点登录客户端，无法注销客户端");
                return;
            }

            for (String clientAddress : sdkProperties.getSso().getClientAddress()) {
                try {
                    authorizationServerHelper.clientLogout(clientAddress, sdkProperties.getSso().getAuthorizeRevokeEndpoint() + "?" + SdkSsoConstants.PARAM_SSO_SERVER + "=true", ticket);
                } catch (Exception e) {
                    log.error("SSO调用客户端注销异常：", e);
                }
            }
        });

    }

    private void writeTicket(HttpServletRequest request, HttpServletResponse response, GeneralUserDetails user) {
        var domains = sdkProperties.getSso().getCookieDomains();
        if (CollectionUtils.isEmpty(domains)) {
            log.warn("未配置域名，无需生成单点登录ticket写入cookie");
            return;
        }

        // 生成ticket
        String ticket = null;
        if (sdkProperties.getServer()) {
            ticket = ticketGenerator.generate(user.getUser());
        } else {
            // 如果不是服务端，需要调用服务端进行生成ticket
            String account = getAccount(user);
            Assert.hasText(account, "账号为空");
            ticket = authorizationServerHelper.generateTicket(sdkProperties.getAuthServer(), account, sdkProperties.getSso().getAccountType().name());
        }

        // 写入ticket
        if (!StringUtils.hasText(ticket)) {
            log.info("未生成ticket，用户：{}", user.getUsername());
            return;
        }
        log.info("用户{}登录，ticket：{},将写入cookie", user.getUsername(), ticket);
        ticketResolver.save(request, response, ticket);
    }

    private String getAccount(GeneralUserDetails userDetails) {
        var user = userDetails.getUser();
        switch (sdkProperties.getSso().getAccountType()) {
            case USER_ID:
                return user.getId().toString();
            case USER_NAME:
                return user.getUsername();
            case EMAIL:
                return user.getEmail();
            case MOBILE:
                return user.getMobile();
            default:
                return user.getUsername();
        }
    }
}
