package com.elitescloud.boot.auth.provider.sso2.support.impl;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.auth.model.OAuthToken;
import com.elitescloud.boot.auth.provider.config.properties.Sso2Properties;
import com.elitescloud.boot.auth.provider.security.grant.InternalAuthenticationGranter;
import com.elitescloud.boot.auth.provider.sso2.support.SsoClientSupportProvider;
import com.elitescloud.boot.auth.sso.common.SdkSsoConstants;
import com.elitescloud.boot.auth.util.AuthorizationUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.RestTemplateFactory;
import com.elitescloud.boot.util.RestTemplateHelper;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpRequest;
import org.springframework.http.client.ClientHttpRequestExecution;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.lang.NonNull;
import org.springframework.security.core.AuthenticationException;
import org.springframework.util.Assert;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;

/**
 * 默认的sso支持实现.
 *
 * @author Kaiser（wang shao）
 * @date 2024/2/20
 */
@Slf4j
public class DefaultSsoClientSupportProvider implements SsoClientSupportProvider {
    private static final String URI_USER_INFO = "/oauth/sso/v2/getUserByTicket";

    protected final Sso2Properties sso2Properties;
    private final InternalAuthenticationGranter internalAuthenticationGranter;
    protected final RestTemplateHelper restTemplateHelper;

    public DefaultSsoClientSupportProvider(Sso2Properties sso2Properties, InternalAuthenticationGranter internalAuthenticationGranter) {
        this.sso2Properties = sso2Properties;
        this.internalAuthenticationGranter = internalAuthenticationGranter;
        this.restTemplateHelper = this.buildRestTemplateHelper(sso2Properties.getClient());
    }

    @Override
    public ApiResult<OAuthToken> ticket2Token(HttpServletRequest request, HttpServletResponse response, String ticket) {
        if (CharSequenceUtil.isBlank(ticket)) {
            return ApiResult.fail("票据ticket为空，请重新登录");
        }

        // 根据ticket获取用户信息
        var userInfo = this.getUserByTicket(ticket);
        if (userInfo == null) {
            return ApiResult.fail("认证失败，请重新登录");
        }

        // 转换认证令牌
        var authenticationToken = this.convertAuthenticationToken(userInfo);
        if (authenticationToken == null) {
            return ApiResult.fail("转换认证令牌失败");
        }

        // 生成token
        OAuthToken token = null;
        try {
            token = internalAuthenticationGranter.authenticate(request, response, authenticationToken);
        } catch (AuthenticationException e) {
            return ApiResult.fail("认证异常，" + e.getMessage());
        }
        return ApiResult.ok(token);
    }

    protected InternalAuthenticationGranter.InternalAuthenticationToken convertAuthenticationToken(SysUserDTO userDTO) {
        var accountType = sso2Properties.getClient().getAccountType();
        switch (accountType) {
            case USER_ID:
                return new InternalAuthenticationGranter.InternalAuthenticationToken(InternalAuthenticationGranter.IdType.USER_ID, userDTO.getId().toString());
            case USER_NAME:
                return new InternalAuthenticationGranter.InternalAuthenticationToken(InternalAuthenticationGranter.IdType.USERNAME, userDTO.getUsername());
            case EMAIL:
                return new InternalAuthenticationGranter.InternalAuthenticationToken(InternalAuthenticationGranter.IdType.EMAIL, userDTO.getEmail());
            case MOBILE:
                return new InternalAuthenticationGranter.InternalAuthenticationToken(InternalAuthenticationGranter.IdType.MOBILE, userDTO.getMobile());
            default:
                throw new BusinessException("暂不支持的认证类型：" + accountType);
        }
    }

    protected SysUserDTO getUserByTicket(String ticket) {
        var uri = this.buildUriOfGetUser(ticket);
        var userResult = restTemplateHelper.exchangeSafely(uri, HttpMethod.GET, null, new TypeReference<ApiResult<SysUserDTO>>() {
        });

        if (userResult == null) {
            throw new BusinessException("认证服务异常，请稍后再试");
        }
        if (userResult.getData() != null) {
            return userResult.getData();
        }
        log.info("根据ticket获取用户失败：{}，{}", ticket, userResult);
        throw new BusinessException(CharSequenceUtil.blankToDefault(userResult.getMsg(), "认证失败，请稍后再试"));
    }

    protected String buildUriOfGetUser(String ticket) {
        return URI_USER_INFO + "?" + SdkSsoConstants.PARAM_SSO_TICKET + "=" + URLEncoder.encode(ticket, StandardCharsets.UTF_8);
    }

    protected RestTemplateHelper buildRestTemplateHelper(Sso2Properties.Client client) {
        Assert.hasText(client.getServerAddr(), "SSO服务端地址为空");

        var restTemplate = RestTemplateFactory.instance(builder -> builder
                .rootUri(client.getServerAddr())
                .additionalInterceptors(new AuthorizationInterceptor(client.getClientId(), client.getClientSecret()))
        );
        return RestTemplateHelper.instance(restTemplate);
    }

    static class AuthorizationInterceptor implements ClientHttpRequestInterceptor {

        private final String token;

        public AuthorizationInterceptor(String clientId, String clientSecret) {
            Assert.hasText(clientId, "客户端ID为空");
            Assert.hasText(clientSecret, "客户端密码为空");
            try {
                this.token = AuthorizationUtil.encodeBasicAuth(clientId, clientSecret);
            } catch (IOException e) {
                throw new IllegalStateException("初始化sso客户端异常", e);
            }
        }

        @NonNull
        @Override
        public ClientHttpResponse intercept(HttpRequest request, @NonNull byte[] body, @NonNull ClientHttpRequestExecution execution) throws IOException {
            HttpHeaders headers = request.getHeaders();

            // 设置token
            headers.add(HttpHeaders.AUTHORIZATION, token);
            return execution.execute(request, body);
        }
    }
}
