package com.elitescloud.boot.auth.provider.config;

import com.elitescloud.boot.auth.config.AuthorizationSdkProperties;
import com.elitescloud.boot.auth.provider.config.properties.Sso2Properties;
import com.elitescloud.boot.auth.provider.security.grant.InternalAuthenticationGranter;
import com.elitescloud.boot.auth.provider.sso2.common.SsoAuthenticationConvert;
import com.elitescloud.boot.auth.provider.sso2.common.SsoUserInfoConvert;
import com.elitescloud.boot.auth.provider.sso2.common.TicketProvider;
import com.elitescloud.boot.auth.provider.sso2.controller.SsoClientSupportController;
import com.elitescloud.boot.auth.provider.sso2.controller.SsoServerSupportController;
import com.elitescloud.boot.auth.provider.sso2.controller.SsoUnifyClientSupportController;
import com.elitescloud.boot.auth.provider.sso2.support.SsoClientSupportProvider;
import com.elitescloud.boot.auth.provider.sso2.support.SsoServerSupportProvider;
import com.elitescloud.boot.auth.provider.sso2.support.SsoUnifyClientSupportProvider;
import com.elitescloud.boot.auth.provider.sso2.support.convert.*;
import com.elitescloud.boot.auth.provider.sso2.support.impl.DefaultSsoClientSupportProvider;
import com.elitescloud.boot.auth.provider.sso2.support.impl.DefaultSsoServerSupportProvider;
import com.elitescloud.boot.auth.provider.sso2.support.impl.DefaultTicketProvider;
import com.elitescloud.boot.auth.provider.sso2.support.impl.SsoUnifyClientSupportProviderImpl;
import com.elitescloud.boot.redis.util.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.ldap.core.LdapTemplate;

import java.util.List;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/2/20
 */
@Slf4j
@EnableConfigurationProperties(Sso2Properties.class)
@Import({Sso2Config.ServerConfig.class, Sso2Config.ClientConfig.class, Sso2Config.LdapConfig.class})
class Sso2Config {
    private final Sso2Properties sso2Properties;

    public Sso2Config(Sso2Properties sso2Properties) {
        this.sso2Properties = sso2Properties;
    }

    @Bean
    @ConditionalOnMissingBean
    public TicketProvider ticketProvider(RedisUtils redisUtils) {
        return new DefaultTicketProvider(redisUtils, sso2Properties);
    }

    @Bean
    public PlainSsoAuthenticationConvert plainSsoAuthenticationConvert() {
        return new PlainSsoAuthenticationConvert();
    }

    @Bean
    public JwtSsoAuthenticationConvert jwtSsoAuthenticationConvert() {
        return new JwtSsoAuthenticationConvert();
    }

    @Bean
    public BasicSsoAuthenticationConvert basicSsoAuthenticationConvert() {
        return new BasicSsoAuthenticationConvert();
    }

    @Bean
    public TicketSsoAuthenticationConvert ticketSsoAuthenticationConvert() {
        return new TicketSsoAuthenticationConvert();
    }

    @Bean
    public OidcSsoAuthenticationConvert oidcSsoAuthenticationConvert() {
        return new OidcSsoAuthenticationConvert();
    }

    @Bean
    public SsoUnifyClientSupportProvider ssoUnifyClientSupportProviderDefault(InternalAuthenticationGranter internalAuthenticationGranter,
                                                                              List<SsoAuthenticationConvert> authenticationConvert) {
        return new SsoUnifyClientSupportProviderImpl(sso2Properties, internalAuthenticationGranter, authenticationConvert);
    }

    @Bean
    public SsoUnifyClientSupportController ssoUnifyClientSupportController(SsoUnifyClientSupportProvider ssoSupportProvider) {
        return new SsoUnifyClientSupportController(ssoSupportProvider);
    }

    @ConditionalOnProperty(prefix = Sso2Properties.CONFIG_PREFIX + ".server", name = "enabled", havingValue = "true")
    class ServerConfig {
        private final Sso2Properties sso2Properties;

        public ServerConfig(Sso2Properties sso2Properties) {
            this.sso2Properties = sso2Properties;
            log.info("启用单点登录服务端");
        }

        @Bean
        @ConditionalOnMissingBean
        public SsoServerSupportProvider defaultSsoServerSupportProvider(TicketProvider ticketProvider,
                                                                        List<SsoUserInfoConvert<?>> userInfoConverts,
                                                                        AuthorizationSdkProperties sdkProperties) {
            return new DefaultSsoServerSupportProvider(sso2Properties, ticketProvider, userInfoConverts, sdkProperties);
        }

        @Bean
        public SsoServerSupportController ssoServerSupportController(SsoServerSupportProvider supportProvider) {
            return new SsoServerSupportController(supportProvider);
        }
    }

    @ConditionalOnProperty(prefix = Sso2Properties.CONFIG_PREFIX + ".client", name = "enabled", havingValue = "true")
    class ClientConfig {
        private final Sso2Properties sso2Properties;

        public ClientConfig(Sso2Properties sso2Properties) {
            this.sso2Properties = sso2Properties;
            log.info("启用单点登录客户端，服务端地址：{}", sso2Properties.getClient().getServerAddr());
        }

        @Bean
        @ConditionalOnMissingBean
        public SsoClientSupportProvider defaultSsoClientSupportProvider(InternalAuthenticationGranter internalAuthenticationGranter) {
            return new DefaultSsoClientSupportProvider(sso2Properties, internalAuthenticationGranter);
        }

        @Bean
        public SsoClientSupportController ssoClientSupportController(SsoClientSupportProvider supportProvider) {
            return new SsoClientSupportController(supportProvider);
        }
    }

    @ConditionalOnClass({LdapTemplate.class})
    class LdapConfig {
        @Bean
        public LdapSsoAuthenticationConvert ldapSsoAuthenticationConvert() {
            return new LdapSsoAuthenticationConvert();
        }
    }
}
