package com.elitescloud.boot.auth.provider.config.system;

import cn.hutool.captcha.AbstractCaptcha;
import cn.hutool.captcha.CaptchaUtil;
import cn.hutool.captcha.generator.RandomGenerator;
import cn.hutool.core.util.RandomUtil;
import com.elitescloud.boot.auth.provider.common.LoginAccountType;
import com.elitescloud.boot.auth.provider.common.LoginDeviceLimitStrategy;
import com.elitescloud.boot.auth.provider.config.system.captcha.CaptchaCharEnum;
import com.elitescloud.boot.auth.provider.config.system.captcha.CaptchaCodeEnum;
import com.wf.captcha.*;
import com.wf.captcha.base.Captcha;
import org.springframework.boot.context.properties.NestedConfigurationProperty;

import java.io.Serializable;
import java.util.*;

/**
 * @author Michael Li
 * @date AC 2021-04-12:14:13
 */
public class LoginProperties implements Serializable {
    private static final long serialVersionUID = 3572914221247822407L;

    /**
     * 验证码
     */
    @NestedConfigurationProperty
    private CaptchaInfo captchaInfo = new CaptchaInfo();

    /**
     * token中的负载
     * <p>
     * 登录时的表单项
     */
    private List<String> tokenPayloads = new ArrayList<>(8);

    /**
     * 是否开启终端限制
     */
    private Boolean terminalLimit = false;

    /**
     * 是否允许自动注册用户
     */
    private Boolean autoRegister = false;

    /**
     * 登录账号类型
     */
    private Set<LoginAccountType> loginAccountTypes = Set.of(LoginAccountType.USERNAME, LoginAccountType.MOBILE);

    /**
     * 登录设备限制策略
     */
    private List<LoginDeviceLimiter> loginDeviceLimiters = new ArrayList<>(0);

    public LoginProperties() {
    }

    public AbstractCaptcha getCaptchaHu() {
        if (Objects.isNull(captchaInfo)) {
            return null;
        }

        if (captchaInfo.getCodeType() == CaptchaCodeEnum.line || captchaInfo.getCodeType() == CaptchaCodeEnum.circle || captchaInfo.getCodeType() == CaptchaCodeEnum.shear) {
            AbstractCaptcha captcha = null;
            switch (captchaInfo.getCodeType()) {
                case line:
                    captcha = CaptchaUtil.createLineCaptcha(captchaInfo.getWidth(), captchaInfo.getHeight());
                    break;
                case circle:
                    captcha = CaptchaUtil.createCircleCaptcha(captchaInfo.getWidth(), captchaInfo.getHeight());
                    break;
                default:
                    captcha = CaptchaUtil.createShearCaptcha(captchaInfo.getWidth(), captchaInfo.getHeight());
                    break;
            }

            String baseStr = null;
            switch (captchaInfo.getCharType()) {
                case NUMBER:
                    baseStr = RandomUtil.BASE_NUMBER;
                    break;
                case CHAR:
                    baseStr = RandomUtil.BASE_CHAR + RandomUtil.BASE_CHAR.toUpperCase();
                    break;
                case CHAR_UPPER:
                    baseStr = RandomUtil.BASE_CHAR.toUpperCase();
                    break;
                case CHAR_LOWER:
                    baseStr = RandomUtil.BASE_CHAR;
                    break;
                case NUMBER_CHAR_UPPER:
                    baseStr = RandomUtil.BASE_CHAR_NUMBER_LOWER.toUpperCase();
                    break;
                default:
                    baseStr = RandomUtil.BASE_CHAR_NUMBER;
                    break;
            }
            captcha.setGenerator(new RandomGenerator(baseStr, captchaInfo.getLength()));
            return captcha;
        }

        return null;
    }

    /**
     * 获取验证码生产类
     *
     * @return /
     */
    public Captcha getCaptcha() {
        if (Objects.isNull(captchaInfo)) {
            captchaInfo = new CaptchaInfo();
            if (Objects.isNull(captchaInfo.getCodeType())) {
                captchaInfo.setCodeType(CaptchaCodeEnum.arithmetic);
            }
            if (Objects.isNull(captchaInfo.getCharType())) {
                captchaInfo.setCharType(CaptchaCharEnum.DEFAULT);
            }
        }
        return switchCaptcha(captchaInfo);
    }

    /**
     * 依据配置信息生产验证码
     *
     * @param captchaInfo 验证码配置信息
     * @return /
     */
    private Captcha switchCaptcha(CaptchaInfo captchaInfo) {
        Captcha captcha;
        synchronized (this) {
            switch (captchaInfo.getCodeType()) {
                case arithmetic:
                    // 算术类型 https://gitee.com/whvse/EasyCaptcha
                    captcha = new ArithmeticCaptcha(captchaInfo.getWidth(), captchaInfo.getHeight());
                    break;
                case chinese:
                    captcha = new ChineseCaptcha(captchaInfo.getWidth(), captchaInfo.getHeight());
                    break;
                case chinese_gif:
                    captcha = new ChineseGifCaptcha(captchaInfo.getWidth(), captchaInfo.getHeight());
                    break;
                case gif:
                    captcha = new GifCaptcha(captchaInfo.getWidth(), captchaInfo.getHeight());
                    break;
                case spec:
                    captcha = new SpecCaptcha(captchaInfo.getWidth(), captchaInfo.getHeight());
                    break;
                default:
                    throw new RuntimeException("验证码配置信息错误！！！正确配置查看 " + CaptchaCodeEnum.class.getName());
            }
        }
        captcha.setLen(captchaInfo.getLength());

        // 字符类型
        switch (captchaInfo.getCharType()) {
            case NUMBER:
                captcha.setCharType(Captcha.TYPE_ONLY_NUMBER);
                break;
            case CHAR:
                captcha.setCharType(Captcha.TYPE_ONLY_CHAR);
                break;
            case CHAR_UPPER:
                captcha.setCharType(Captcha.TYPE_ONLY_UPPER);
                break;
            case CHAR_LOWER:
                captcha.setCharType(Captcha.TYPE_ONLY_LOWER);
                break;
            case NUMBER_CHAR_UPPER:
                captcha.setCharType(Captcha.TYPE_NUM_AND_UPPER);
                break;
            default:
                captcha.setCharType(Captcha.TYPE_DEFAULT);
        }
        return captcha;
    }

    public List<String> getTokenPayloads() {
        return tokenPayloads;
    }

    public void setTokenPayloads(List<String> tokenPayloads) {
        this.tokenPayloads = tokenPayloads;
    }

    public CaptchaInfo getCaptchaInfo() {
        return captchaInfo;
    }

    public void setCaptchaInfo(CaptchaInfo captchaInfo) {
        this.captchaInfo = captchaInfo;
    }

    public Boolean getTerminalLimit() {
        return terminalLimit;
    }

    public void setTerminalLimit(Boolean terminalLimit) {
        this.terminalLimit = terminalLimit;
    }

    public Boolean getAutoRegister() {
        return autoRegister;
    }

    public void setAutoRegister(Boolean autoRegister) {
        this.autoRegister = autoRegister;
    }

    public Set<LoginAccountType> getLoginAccountTypes() {
        return loginAccountTypes;
    }

    public void setLoginAccountTypes(Set<LoginAccountType> loginAccountTypes) {
        this.loginAccountTypes = loginAccountTypes;
    }

    public List<LoginDeviceLimiter> getLoginDeviceLimiters() {
        return loginDeviceLimiters;
    }

    public void setLoginDeviceLimiters(List<LoginDeviceLimiter> loginDeviceLimiters) {
        this.loginDeviceLimiters = loginDeviceLimiters;
    }

    public static class LoginDeviceLimiter {

        /**
         * 客户端ID
         */
        private String clientId;

        /**
         * 登录设备限制策略
         */
        private LoginDeviceLimitStrategy strategy = LoginDeviceLimitStrategy.NO_LIMIT;

        /**
         * 一个设备限制提示语
         */
        private String oneLimitTip = "您已在其它设备登录";

        public String getClientId() {
            return clientId;
        }

        public void setClientId(String clientId) {
            this.clientId = clientId;
        }

        public LoginDeviceLimitStrategy getStrategy() {
            return strategy;
        }

        public void setStrategy(LoginDeviceLimitStrategy strategy) {
            this.strategy = strategy;
        }

        public String getOneLimitTip() {
            return oneLimitTip;
        }

        public void setOneLimitTip(String oneLimitTip) {
            this.oneLimitTip = oneLimitTip;
        }
    }
}
