package com.elitescloud.boot.auth.provider.provider;

import cn.hutool.captcha.AbstractCaptcha;
import cn.hutool.core.util.IdUtil;
import com.elitescloud.boot.auth.client.tool.RedisHelper;
import com.elitescloud.boot.auth.provider.config.system.CaptchaEnum;
import com.elitescloud.boot.auth.provider.config.system.Constants;
import com.elitescloud.boot.auth.provider.resp.CaptchaRespVO;
import com.elitescloud.boot.exception.BusinessException;
import com.wf.captcha.base.Captcha;
import lombok.extern.log4j.Log4j2;
import org.springframework.util.Assert;

import java.time.Duration;
import java.util.concurrent.TimeUnit;

/**
 * 验证码相关接口.
 *
 * @author Kaiser（wang shao）
 * @date 2022/01/10
 */
@Log4j2
public class CaptchaProvider {

    private final RedisHelper redisHelper;

    public CaptchaProvider(RedisHelper redisHelper) {
        this.redisHelper = redisHelper;
    }

    /**
     * 生成验证码
     *
     * @param captchaEnum 验证码类型
     * @param captchaHu   验证码信息
     * @param ttl         验证码有效期
     * @return 验证码
     */
    public CaptchaRespVO generateCaptchaHu(CaptchaEnum captchaEnum, AbstractCaptcha captchaHu, Duration ttl) {
        Assert.notNull(captchaEnum, "未知验证码类型");
        Assert.notNull(captchaHu, "验证码为空");
        String uuid = IdUtil.simpleUUID();
        String key = buildCacheKey(captchaEnum, uuid);

        CaptchaRespVO respVO = new CaptchaRespVO();
        respVO.setUuid(uuid);

        String code = captchaHu.getCode();
        cacheCaptcha(key, code, ttl);

        String imgData = captchaHu.getImageBase64Data();
        respVO.setImg(imgData.startsWith("data:") ? imgData : "data:image/jpg;base64," + imgData);
        respVO.setCaptext(code);
        return respVO;
    }

    /**
     * 生成验证码
     *
     * @param captchaEnum 验证码类型
     * @param captcha     验证码信息
     * @param ttl         验证码有效期
     * @return 验证码
     */
    public CaptchaRespVO generateCaptcha(CaptchaEnum captchaEnum, Captcha captcha, Duration ttl) {
        Assert.notNull(captchaEnum, "未知验证码类型");
        Assert.notNull(captcha, "验证码为空");
        String uuid = IdUtil.simpleUUID();
        String key = buildCacheKey(captchaEnum, uuid);

        CaptchaRespVO respVO = new CaptchaRespVO();
        respVO.setUuid(uuid);

        String code = captcha.text();
        cacheCaptcha(key, code, ttl);

        respVO.setImg(captcha.toBase64());
        respVO.setCaptext(code);
        return respVO;
    }

    /**
     * 校验验证码
     *
     * @param captchaEnum 验证码类型
     * @param uuid        验证码uuid
     * @param text        验证码内容
     * @return 是否校验通过
     */
    public boolean verifyCaptcha(CaptchaEnum captchaEnum, String uuid, String text) {
        Assert.notNull(captchaEnum, "未知验证码类型");
        Assert.hasText(uuid, "未知校验码标识");
        Assert.hasText(text, "未知校验码");
        String key = buildCacheKey(captchaEnum, uuid);

        String captcha = null;
        try {
            captcha = (String) redisHelper.execute(redisUtils -> redisUtils.get(key));
        } catch (Exception e) {
            throw new BusinessException("验证验证码失败", e);
        }
        if (text.equalsIgnoreCase(captcha)) {
            // 校验通过
            try {
                redisHelper.execute(redisUtils -> {
                    redisUtils.del(key);
                    return null;
                });
            } catch (Exception e) {
                log.error("验证码【" + key + "】校验通过后，删除验证码失败：", e);
            }
            return true;
        }
        return false;
    }

    private String buildCacheKey(CaptchaEnum captchaEnum, String uuid) {
        return Constants.CACHE_KEY_PREFIX + "CAPTCHA:" + captchaEnum + ":" + uuid;
    }

    private void cacheCaptcha(String key, String value, Duration ttl) {
        try {
            redisHelper.execute(redisUtils -> {
                if (ttl == null) {
                    redisUtils.set(key, value);
                    return null;
                }

                redisUtils.set(key, value, ttl.getSeconds(), TimeUnit.SECONDS);
                return null;
            });
        } catch (Exception e) {
            throw new BusinessException("生成验证码失败", e);
        }
    }
}
