package com.elitescloud.boot.auth.provider.provider.wechat;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjUtil;
import com.elitescloud.boot.auth.client.tool.RedisHelper;
import com.elitescloud.boot.auth.provider.common.WechatAppProvider;
import com.elitescloud.boot.auth.provider.common.param.WechatAppTypeEnum;
import com.elitescloud.boot.auth.provider.provider.wechat.param.*;
import com.elitescloud.boot.exception.BusinessException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.lang.NonNull;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.function.Function;

/**
 * 微信工具.
 *
 * @author Kaiser（wang shao）
 * @date 2/14/2023
 */
public class WechatTemplate {
    private static final Logger logger = LoggerFactory.getLogger(WechatTemplate.class);

    private final WechatAppProvider wechatAppProvider;
    private final RedisHelper redisHelper;
    private final WechatTool wechatTool;

    public WechatTemplate(@NotNull WechatAppProvider wechatAppProvider,
                          @NotNull RedisHelper redisHelper) {
        this.wechatAppProvider = wechatAppProvider;
        this.redisHelper = redisHelper;
        this.wechatTool = WechatTool.getInstance();
    }

    /**
     * 获取手机号（小程序）
     *
     * @param appId 应用ID
     * @param code  授权码
     * @return 手机号信息
     */
    public WechatPhoneInfo getPhoneInfo(@NotBlank String appId, @NotBlank String code) {
        Assert.hasText(appId, "微信应用ID为空");
        Assert.hasText(code, "微信授权码为空");

        return this.executeWithToken(appId, token -> wechatTool.getPhoneNumber(token, code));
    }

    /**
     * 获取openId（小程序）
     *
     * @param appId  应用ID
     * @param jsCode 授权码
     * @return openId
     */
    public WechatCode2Session getCode2Session(@NotBlank String appId, @NotBlank String jsCode) {
        Assert.hasText(appId, "微信应用ID为空");
        Assert.hasText(jsCode, "微信授权码为空");

        // 获取微信app信息
        var app = wechatAppProvider.getApp(appId);
        if (app == null) {
            throw new BusinessException("未知微信应用");
        }

        return wechatTool.jscode2Session(appId, app.getSecret(), jsCode);
    }

    /**
     * 授权码获取用户信息
     *
     * @param appId
     * @param code
     * @return
     */
    public WechatUserInfo getUserInfo(@NonNull String appId, @NonNull String code) {
        // 获取微信app信息
        var app = wechatAppProvider.getApp(appId);
        if (app == null) {
            throw new BusinessException("未知微信应用");
        }

        // code换取access_token
        WechatAccessToken accessToken = wechatTool.oauth2AccessToken(appId, app.getSecret(), code);
        if (accessToken == null) {
            throw new BusinessException("授权或已过期，请重新操作");
        }

        // 获取用户信息
        return wechatTool.snsUserInfo(accessToken.getAccessToken(), accessToken.getOpenId());
    }

    /**
     * 获取OpenId
     *
     * @param appId 应用ID
     * @param code  授权码
     * @return openId
     */
    public WechatOpenIdAndUnionId getOpenId(@NonNull String appId, @NotBlank String code) {
        // 获取微信app信息
        var app = wechatAppProvider.getApp(appId);
        if (app == null) {
            throw new BusinessException("未知微信应用");
        }

        WechatAppTypeEnum appType = ObjUtil.defaultIfNull(app.getAppType(), WechatAppTypeEnum.APPLET);
        switch (appType) {
            case APPLET: {
                WechatCode2Session code2Session = getCode2Session(app.getAppid(), code);
                return new WechatOpenIdAndUnionId(code2Session.getOpenid(), code2Session.getUnionid());
            }
            default:
                var userInfo = getUserInfo(app.getAppid(), code);
                return new WechatOpenIdAndUnionId(userInfo.getOpenid(), userInfo.getUnionid());
        }
    }

    private <T extends BaseWechatResult> T executeWithToken(@NotBlank String appId, @NotNull Function<String, T> supplier) {
        // 获取token
        String accessToken = getAccessToken(appId);
        T result = supplier.apply(accessToken);
        if (result.isSuccess()) {
            return result;
        }

        // 刷新token重试
        clearAccessToken(appId);
        result = supplier.apply(accessToken);
        if (!result.isSuccess()) {
            logger.error("调用微信接口失败：{}", result);
        }

        return result;
    }

    private String getAccessToken(String appId) {
        // 先从缓存获取
        String key = "wechat:accesstoken:" + appId;
        String accessToken = (String) redisHelper.execute(redisUtils -> redisUtils.get(key));
        if (StringUtils.hasText(accessToken)) {
            return accessToken;
        }

        // 调用微信接口生成token
        var app = wechatAppProvider.getApp(appId);
        Assert.notNull(app, appId + "微信账号不存在");
        String appSecret = app.getSecret();
        Assert.hasText(appSecret, appId + "微信账号密码为空");
        var tokenResult = wechatTool.getAccessToken(appId, appSecret);
        if (tokenResult == null || CharSequenceUtil.isBlank(tokenResult.getAccessToken())) {
            throw new BusinessException("微信授权异常，请稍后再试");
        }

        // 将token缓存
        redisHelper.execute(redisUtils -> redisUtils.set(key, tokenResult.getAccessToken(), tokenResult.getExpiresIn() - 60));
        return tokenResult.getAccessToken();
    }

    private void clearAccessToken(String appId) {
        String key = "wechat:accesstoken:" + appId;
        redisHelper.execute(redisUtils -> {
            redisUtils.del(key);
            return null;
        });
    }
}
