package com.elitescloud.boot.auth.provider.security.grant.alipay_phone;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.auth.client.common.AuthorizationException;
import com.elitescloud.boot.auth.provider.provider.alipay.AlipayTemplate;
import com.elitescloud.boot.auth.provider.provider.wechat.WechatTemplate;
import com.elitescloud.boot.auth.provider.security.grant.AbstractCustomAuthenticationProvider;
import com.elitescloud.boot.auth.provider.security.grant.wechat_phone.WechatPhoneAuthenticationToken;
import com.elitescloud.boot.auth.provider.security.jackson.mixin.grant.MixinAlipayPhoneAuthenticationToken;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.AuthenticationException;
import org.springframework.util.StringUtils;

/**
 * 支付宝手机号的身份认证.
 *
 * @author Kaiser（wang shao）
 * @date 2022/01/01
 */
@Log4j2
public class AlipayPhoneAuthenticationProvider extends AbstractCustomAuthenticationProvider<AlipayPhoneAuthenticationToken> {

    @Autowired
    private AlipayTemplate alipayTemplate;

    @Override
    protected GeneralUserDetails retrieveUser(AlipayPhoneAuthenticationToken authentication) throws AuthenticationException {
        String appId = (String) authentication.getPrincipal();
        if (!StringUtils.hasText(appId)) {
            throw new AuthorizationException("应用ID为空");
        }
        String resp = (String) authentication.getCredentials();
        if (!StringUtils.hasText(resp)) {
            throw new AuthorizationException("支付宝授权内容为空");
        }
        if (authentication.isAutoRegister() && Boolean.FALSE.equals(configProperties.getLogin().getAutoRegister())) {
            throw new AuthorizationException("不支持自动注册账号");
        }

        // 调用微信获取手机号
        String phone = null;
        try {
            phone = alipayTemplate.getPhoneNumber(appId, resp);
        } catch (Exception e) {
            String msg = e instanceof IllegalArgumentException || e instanceof BusinessException ? ", " + e.getMessage() : "";
            throw new AuthorizationException("支付宝授权获取手机号异常" + msg, e);
        }
        if (CharSequenceUtil.isBlank(phone)) {
            throw new AuthorizationException("支付宝未绑定手机号");
        }

        var user = authentication.isAutoRegister() ? userDetailManager.registerOnLoadUserByMobileNotFound(phone) :
                userDetailManager.loadUserByMobile(phone);
        if (user == null) {
            throw new AuthorizationException("手机号错误或账号不存在");
        }
        return user;
    }

    @Override
    public Class<AlipayPhoneAuthenticationToken> getAuthenticationTokenType() {
        return AlipayPhoneAuthenticationToken.class;
    }

    @Override
    public Class<?> getMixinAuthenticationTokenType() {
        return MixinAlipayPhoneAuthenticationToken.class;
    }
}
