package com.elitescloud.boot.auth.provider.security.grant.ldap;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.auth.client.common.AuthorizationException;
import com.elitescloud.boot.auth.provider.config.properties.LdapProperties;
import com.elitescloud.boot.auth.provider.security.grant.AbstractCustomAuthenticationProvider;
import com.elitescloud.boot.auth.provider.security.jackson.mixin.grant.MixinLdapAuthenticationToken;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.PropertyMapper;
import org.springframework.ldap.core.LdapTemplate;
import org.springframework.ldap.core.support.LdapContextSource;
import org.springframework.ldap.filter.AndFilter;
import org.springframework.ldap.filter.EqualsFilter;
import org.springframework.security.core.AuthenticationException;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * LDAP身份认证.
 *
 * @author Kaiser（wang shao）
 * @date 2024/10/7
 */
public class LdapAuthenticationProvider extends AbstractCustomAuthenticationProvider<LdapAuthenticationToken> implements InitializingBean {
    private static final Logger logger = LoggerFactory.getLogger(LdapAuthenticationProvider.class);

    private LdapProperties ldapProperties;
    private LdapTemplate ldapTemplate;

    @Override
    protected GeneralUserDetails retrieveUser(LdapAuthenticationToken authentication) throws AuthenticationException {
        var authenticated = this.authenticateByLdap(authentication);
        if (!authenticated) {
            throw new AuthorizationException(CharSequenceUtil.blankToDefault(ldapProperties.getAuthenticatedFailMsg(), "LDAP认证失败"));
        }

        switch (ldapProperties.getLoginAccountType()) {
            case ID:
                return userDetailManager.loadUserById((String) authentication.getPrincipal());
            case USERNAME:
                return userDetailManager.loadUserByUsername((String) authentication.getPrincipal());
            case MOBILE:
                return userDetailManager.loadUserByMobile((String) authentication.getPrincipal());
            case EMAIL:
                return userDetailManager.loadUserByEmail((String) authentication.getPrincipal());
            default:
                throw new AuthorizationException("认证失败，暂不支持的账号类型");
        }
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        if (ldapProperties == null || !ldapProperties.isEnabled()) {
            return;
        }
        logger.info("LDAP is enabled, the server is：{}", Arrays.toString(ldapProperties.getUrls()));

        this.initLdapTemplate();
    }

    @Autowired
    public void setLdapProperties(LdapProperties ldapProperties) {
        this.ldapProperties = ldapProperties;
    }

    @Override
    public Class<LdapAuthenticationToken> getAuthenticationTokenType() {
        return LdapAuthenticationToken.class;
    }

    @Override
    public Class<?> getMixinAuthenticationTokenType() {
        return MixinLdapAuthenticationToken.class;
    }

    private boolean authenticateByLdap(LdapAuthenticationToken authentication) {
        if (CharSequenceUtil.isBlank((String) authentication.getPrincipal())) {
            throw new AuthorizationException("账号为空");
        }

        // 通用条件
        Map<String, Object> attributes = new HashMap<>(8);
        if (CollUtil.isNotEmpty(ldapProperties.getLoginAttributes())) {
            attributes.putAll(ldapProperties.getLoginAttributes());
        }
        if (CollUtil.isNotEmpty(authentication.getAttributes())) {
            attributes.putAll(authentication.getAttributes());
        }

        AndFilter filter = new AndFilter().and(new EqualsFilter(ldapProperties.getLoginAttributeName(), (String) authentication.getPrincipal()));
        for (Map.Entry<String, Object> entry : attributes.entrySet()) {
            filter.and(entry.getValue() instanceof Integer ? new EqualsFilter(entry.getKey(), (Integer) entry.getValue()) : new EqualsFilter(entry.getKey(), entry.getValue().toString()));
        }

        try {
            return ldapTemplate.authenticate(ldapProperties.getBase(), filter.encode(), (String) authentication.getCredentials());
        } catch (Exception e) {
            throw new AuthorizationException("LDAP认证失败," + e.getMessage(), e);
        }
    }

    private void initLdapTemplate() {
        var contextSource = this.buildLdapContextSource();
        LdapProperties.Template template = ldapProperties.getTemplate();
        PropertyMapper propertyMapper = PropertyMapper.get().alwaysApplyingWhenNonNull();
        ldapTemplate = new LdapTemplate(contextSource);
        propertyMapper.from(template.isIgnorePartialResultException())
                .to(ldapTemplate::setIgnorePartialResultException);
        propertyMapper.from(template.isIgnoreNameNotFoundException()).to(ldapTemplate::setIgnoreNameNotFoundException);
        propertyMapper.from(template.isIgnoreSizeLimitExceededException())
                .to(ldapTemplate::setIgnoreSizeLimitExceededException);
    }

    private LdapContextSource buildLdapContextSource() {
        Assert.notEmpty(ldapProperties.getUrls(), "LDAP url未配置");

        LdapContextSource source = new LdapContextSource();
        PropertyMapper propertyMapper = PropertyMapper.get().alwaysApplyingWhenNonNull();
        propertyMapper.from(ldapProperties.getUsername()).to(source::setUserDn);
        propertyMapper.from(ldapProperties.getPassword()).to(source::setPassword);
        propertyMapper.from(ldapProperties.getAnonymousReadOnly()).to(source::setAnonymousReadOnly);
        propertyMapper.from(ldapProperties.getBase()).to(source::setBase);
        propertyMapper.from(ldapProperties.getUrls()).to(source::setUrls);
        propertyMapper.from(ldapProperties.getBaseEnvironment())
                .to((baseEnvironment) -> source.setBaseEnvironmentProperties(Collections.unmodifiableMap(baseEnvironment)));

        return source;
    }

}
