package com.elitescloud.boot.auth.provider.security.handler;

import com.elitescloud.boot.auth.client.common.AuthorizationException;
import com.elitescloud.boot.auth.client.config.security.handler.AbstractHandler;
import com.elitescloud.boot.auth.client.config.support.AuthenticationCallable;
import com.elitescloud.boot.auth.provider.common.AuthorizationConstant;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import lombok.extern.log4j.Log4j2;
import org.springframework.http.HttpStatus;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.oauth2.server.resource.InvalidBearerTokenException;
import org.springframework.security.web.authentication.AuthenticationFailureHandler;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

/**
 * 认证失败的处理.
 * <p>
 * 针对授权已过期和认证失败的处理
 *
 * @author Kaiser（wang shao）
 * @date 2022/6/21
 */
@Log4j2
public class DefaultAuthenticationFailureHandler extends AbstractHandler implements AuthenticationFailureHandler {

    private final AuthenticationCallable authenticationCallable;

    public DefaultAuthenticationFailureHandler(AuthenticationCallable authenticationCallable) {
        this.authenticationCallable = authenticationCallable;
    }

    @Override
    public void onAuthenticationFailure(HttpServletRequest request, HttpServletResponse response, AuthenticationException exception) throws IOException, ServletException {
        String errorMessage = exception.getMessage();
        if (exception instanceof InvalidBearerTokenException) {
            errorMessage = "身份认证无效或已过期，请重新登录";
        } else {
            log.error("认证失败：", exception);
        }

        ApiCode apiCode = exception instanceof AuthorizationException ? ((AuthorizationException) exception).getApiCode() : ApiCode.AUTHENTICATION_EXCEPTION;

        var result = ApiResult.result(apiCode, errorMessage, null);
        if (apiCode == ApiCode.UNAUTHORIZED) {
            // 需要重新登录
            writeResponse(response, result, HttpStatus.UNAUTHORIZED);
        } else {
            writeResponse(response, result);
        }

        // 回调处理
        request.setAttribute(AuthorizationConstant.REQUEST_ATTRIBUTE_LOGIN_RESULT, result);
        if (authenticationCallable != null) {
            authenticationCallable.onLoginFailure(request, response, null, exception);
        }
    }
}
