package com.elitescloud.boot.auth.provider.sso2.support.convert;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.auth.provider.security.grant.InternalAuthenticationGranter;
import com.elitescloud.boot.auth.provider.sso2.common.SsoConvertProperty;
import com.elitescloud.boot.auth.provider.sso2.common.SsoTypeEnum;
import com.elitescloud.boot.auth.provider.sso2.support.convert.properties.TicketSsoConvertProperty;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.boot.util.RestTemplateFactory;
import com.elitescloud.boot.util.RestTemplateHelper;
import com.fasterxml.jackson.core.type.TypeReference;
import org.jetbrains.annotations.Nullable;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.web.util.UriComponentsBuilder;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.HashMap;
import java.util.Map;

/**
 * ticket类型.
 *
 * @author Kaiser（wang shao）
 * @date 2025/6/8 周日
 */
public class TicketSsoAuthenticationConvert extends BasePlainSsoAuthenticationConvert {

    private final RestTemplateHelper restTemplateHelper;

    public TicketSsoAuthenticationConvert() {
        this.restTemplateHelper = RestTemplateHelper.instance(RestTemplateFactory.instance());
    }

    @Override
    public SsoTypeEnum supportType() {
        return SsoTypeEnum.TICKET;
    }

    @Override
    public <T extends SsoConvertProperty> Class<T> propertyType() {
        return (Class<T>) TicketSsoConvertProperty.class;
    }

    @Nullable
    @Override
    public <T extends SsoConvertProperty> InternalAuthenticationGranter.InternalAuthenticationToken convert(HttpServletRequest request, HttpServletResponse response, T properties) {
        TicketSsoConvertProperty props = (TicketSsoConvertProperty) properties;

        String value = getParam(request, props.getParamName(), props.getParamIn());
        if (CharSequenceUtil.isBlank(value)) {
            throw new IllegalArgumentException("参数为空:" + props.getParamName());
        }

        String username = getUsername(value, props);
        if (CharSequenceUtil.isBlank(username)) {
            throw new BusinessException("授权账户为空");
        }
        return new InternalAuthenticationGranter.InternalAuthenticationToken(props.getIdType(), username);
    }

    private String getUsername(String ticket, TicketSsoConvertProperty props) {
        Assert.notBlank(props.getTicketParamName(), "ticket参数名称为空");

        // 查询用户信息
        String url = props.getUserInfoEndpoint();
        HttpHeaders headers = new HttpHeaders();
        Map<String, Object> body = new HashMap<>();
        switch (props.getUserInfoParamIn()) {
            case QUERY:
                url = expendUrl(url, ticket, props);
                headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);
                break;
            case FORM:
                expandBody(body, ticket, props);
                headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);
                break;
            case BODY:
                expandBody(body, ticket, props);
                headers.setContentType(MediaType.APPLICATION_JSON);
                break;
            default:
                throw new IllegalArgumentException("不支持的参数位置:" + props.getUserInfoParamIn());
        }
        String resp = restTemplateHelper.exchange(url, props.getUserInfoMethod(), new HttpEntity<>(body, headers), new ParameterizedTypeReference<>() {
        });
        if (CharSequenceUtil.isBlank(resp) || CharSequenceUtil.isBlank(props.getUserInfoParamPath())) {
            return resp;
        }

        Map<String, Object> jsonResp = JSONUtil.json2Obj(resp, new TypeReference<>() {}, true, () -> "解析用户信息响应内容异常");
        return getValueByPath(props.getUserInfoParamPath(), jsonResp);
    }

    private String expendUrl(String url, String ticket, TicketSsoConvertProperty props) {
        UriComponentsBuilder urlBuilder = UriComponentsBuilder.fromHttpUrl(url)
                .queryParam(props.getTicketParamName(), ticket);

        if (CollUtil.isNotEmpty(props.getUserInfoAttributeMap())) {
            props.getUserInfoAttributeMap().forEach(urlBuilder::queryParam);
        }

        return urlBuilder.toUriString();
    }

    private void expandBody(Map<String, Object> body, String ticket, TicketSsoConvertProperty props) {
        body.put(props.getTicketParamName(), ticket);

        if (CollUtil.isNotEmpty(props.getUserInfoAttributeMap())) {
            body.putAll(props.getUserInfoAttributeMap());
        }
    }
}
