package com.elitescloud.boot.auth.provider.cas.support;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.auth.cas.model.AuthUserDTO;
import com.elitescloud.boot.auth.cas.provider.OAuth2ClientTemplate;
import com.elitescloud.boot.auth.cas.provider.UserTransferHelper;
import com.elitescloud.boot.auth.client.common.AuthorizationException;
import com.elitescloud.boot.auth.client.common.AuthorizationType;
import com.elitescloud.boot.auth.client.config.AuthorizationProperties;
import com.elitescloud.boot.auth.client.config.security.resolver.BearerTokenResolver;
import com.elitescloud.boot.auth.client.config.security.resolver.impl.DefaultBearerTokenResolver;
import com.elitescloud.boot.auth.config.AuthorizationSdkProperties;
import com.elitescloud.boot.auth.model.OAuthToken;
import com.elitescloud.boot.auth.provider.cas.AuthorizeSettingCustomizer;
import com.elitescloud.boot.auth.provider.cas.CasUserResolver;
import com.elitescloud.boot.auth.provider.cas.OidcUserResolver;
import com.elitescloud.boot.auth.provider.cas.model.AuthorizeSettingVO;
import com.elitescloud.boot.auth.provider.cas.model.OidcUser;
import com.elitescloud.boot.auth.provider.common.AuthorizationConstant;
import com.elitescloud.boot.auth.provider.security.grant.InternalAuthenticationGranter;
import com.elitescloud.boot.auth.util.AuthCompatibilityUtil;
import com.elitescloud.boot.constant.AuthenticationClaim;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.boot.util.JwtUtil;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.oauth2.core.OAuth2AuthenticationException;
import org.springframework.security.oauth2.jwt.Jwt;
import org.springframework.security.oauth2.jwt.JwtDecoder;
import org.springframework.security.oauth2.jwt.JwtException;
import org.springframework.util.Assert;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.Collections;
import java.util.Map;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/19
 */
@Log4j2
public class CasLoginSupportProvider {
    private final AuthorizationProperties authorizationProperties;
    private final AuthorizationSdkProperties sdkProperties;
    private final OAuth2ClientTemplate oAuth2ClientTemplate;
    private final InternalAuthenticationGranter internalAuthenticationGranter;
    private final UserTransferHelper userTransferHelper;

    private ObjectMapper objectMapper;
    private JwtDecoder jwtDecoder;
    private OidcUserResolver oidcUserResolver;
    private CasUserResolver casUserResolver;
    private AuthorizeSettingCustomizer authorizeSettingCustomizer;
    private final BearerTokenResolver tokenResolver = new DefaultBearerTokenResolver();

    public CasLoginSupportProvider(AuthorizationProperties authorizationProperties, AuthorizationSdkProperties sdkProperties,
                                   OAuth2ClientTemplate oAuth2ClientTemplate, InternalAuthenticationGranter internalAuthenticationGranter) {
        this.authorizationProperties = authorizationProperties;
        this.sdkProperties = sdkProperties;
        this.oAuth2ClientTemplate = oAuth2ClientTemplate;
        this.internalAuthenticationGranter = internalAuthenticationGranter;
        this.userTransferHelper = UserTransferHelper.getInstance(sdkProperties.getAuthServer());
    }

    /**
     * 获取配置信息
     *
     * @return 配置信息
     */
    public ApiResult<AuthorizeSettingVO> getSetting(HttpServletRequest request, HttpServletResponse response) {
        boolean enabled = ObjectUtil.defaultIfNull(sdkProperties.getCasClient().getEnabled(), false);
        AuthorizeSettingVO settingVO = new AuthorizeSettingVO();
        settingVO.setEnabled(enabled);

        if (enabled) {
            var authorizeDTO = oAuth2ClientTemplate.generateAuthorizeInfo(request, response);
            settingVO.setAuthorizeUrl(authorizeDTO.getUrl());
            settingVO.setAuthServer(authorizeDTO.getAuthServer());
            settingVO.setLogoutUrl(authorizeDTO.getLogoutUrl());
        }

        // 自定义设置
        if (authorizeSettingCustomizer != null) {
            authorizeSettingCustomizer.customize(settingVO);
        }

        return ApiResult.ok(settingVO);
    }

    /**
     * 授权码换取token
     *
     * @param code 授权码
     * @return token信息
     */
    public ApiResult<OAuthToken> code2Token(HttpServletRequest request, HttpServletResponse response, @NotBlank String code) {
        if (oAuth2ClientTemplate == null) {
            return ApiResult.fail("未启用统一身份认证");
        }

        // 授权码换取token
        OAuthToken tokenResult = null;
        try {
            tokenResult = oAuth2ClientTemplate.code2Token(request, response, code);
        } catch (OAuth2AuthenticationException e) {
            throw new AuthorizationException(ApiCode.UNAUTHORIZED, "认证超时，请重试", e);
        }
        if (tokenResult == null) {
            return ApiResult.fail("认证失败，请稍后再试");
        }


        // token换取用户信息
        InternalAuthenticationGranter.InternalAuthenticationToken authenticationToken = null;
        // 获取认证信息
        var userResult = oAuth2ClientTemplate.getUserInfo(request, tokenResult.getTokenType(), tokenResult.getAccessToken());
        if (userResult == null) {
            return ApiResult.fail("认证失败，未获取到用户信息，请稍后再试");
        }

        if (oidcUserResolver != null) {
            OidcUser oidcUser = null;
            try {
                oidcUser = objectMapper.convertValue(userResult, OidcUser.class);
            } catch (IllegalArgumentException e) {
                log.error("账号信息转换异常：", e);
                throw new BusinessException("认证失败，请稍后再试");
            }

            // 内部认证，生成token
            authenticationToken = oidcUserResolver.resolve(oidcUser);
        } else {
            authenticationToken = new InternalAuthenticationGranter.InternalAuthenticationToken(InternalAuthenticationGranter.IdType.USERNAME, userResult.getUsername());
        }

        // 解析客户端ID
        var jwt = decodeToken(tokenResult);
        String clientId = (String) jwt.get(AuthenticationClaim.KEY_CLIENT_ID);
        request.setAttribute(AuthorizationConstant.REQUEST_ATTRIBUTE_CLIENT_ID, CharSequenceUtil.blankToDefault(clientId, sdkProperties.getCasClient().getOauth2Client().getClientId()));

        return this.grantToken(request, response, authenticationToken);
    }

    /**
     * token换取token
     * <p>
     * 支持非web端进行统一身份认证
     *
     * @param request 请求
     * @param token   token
     * @return token
     */
    public ApiResult<OAuthToken> token2Token(HttpServletRequest request, HttpServletResponse response, String token) {
        if (CharSequenceUtil.isBlank(token)) {
            token = tokenResolver.resolve(request);
        }
        if (CharSequenceUtil.isBlank(token)) {
            return ApiResult.fail("认证失败，未发现有效token");
        }

        // 认证授权
        InternalAuthenticationGranter.InternalAuthenticationToken authenticationToken = null;
        if (casUserResolver == null) {
            // 解析用户
            var username = this.obtainUsername(token);
            authenticationToken = new InternalAuthenticationGranter.InternalAuthenticationToken(InternalAuthenticationGranter.IdType.USERNAME, username);
        } else {
            // 解析用户
            var casUserId = this.obtainUserId(token);
            var userInfo = this.queryUserById(casUserId);
            if (userInfo == null) {
                return ApiResult.fail("认证失败，用户账号不存在");
            }
            var userId = this.resolveUserId(userInfo);
            if (userId == null) {
                return ApiResult.fail("认证失败，请确认用户账号存在");
            }
            authenticationToken = new InternalAuthenticationGranter.InternalAuthenticationToken(InternalAuthenticationGranter.IdType.USER_ID, userId.toString());
        }

        // 解析客户端ID
        var jwt = jwtDecoder.decode(token);
        String clientId = jwt.getClaimAsString(AuthenticationClaim.KEY_CLIENT_ID);
        request.setAttribute(AuthorizationConstant.REQUEST_ATTRIBUTE_CLIENT_ID, clientId);

        return this.grantToken(request, response, authenticationToken);
    }

    private Map<String, Object> decodeToken(OAuthToken token) {
        try {
            return JwtUtil.decode(CharSequenceUtil.blankToDefault(token.getIdToken(), token.getAccessToken()));
        } catch (Exception e) {
            log.warn("token解析异常：{}", JSONUtil.toJsonString(token), e);
            return Collections.emptyMap();
        }
    }

    @SuppressWarnings({"deprecation", "removal"})
    private Long resolveUserId(AuthUserDTO userDTO) {
        if (casUserResolver instanceof com.elitescloud.cloudt.authorization.api.provider.cas.CasUserResolver) {
            // 兼容老的
            var user = AuthCompatibilityUtil.convert(userDTO);
            return ((com.elitescloud.cloudt.authorization.api.provider.cas.CasUserResolver) casUserResolver).resolve(user);
        }

        return casUserResolver.execute(userDTO);
    }

    private ApiResult<OAuthToken> grantToken(HttpServletRequest request, HttpServletResponse response,
                                             InternalAuthenticationGranter.InternalAuthenticationToken authenticationToken) {
        OAuthToken token = null;
        try {
            token = internalAuthenticationGranter.authenticate(request, response, authenticationToken);
        } catch (AuthenticationException e) {
            return ApiResult.fail("认证异常，" + e.getMessage());
        }
        return ApiResult.ok(token);
    }

    private AuthUserDTO queryUserByUsername(@NotBlank String username) {
        Assert.hasText(username, "用户账号为空");
        return userTransferHelper.getUserByUsername(username).getData();
    }

    private AuthUserDTO queryUserById(@NotNull Long id) {
        Assert.notNull(id, "用户ID为空");
        return userTransferHelper.getUser(id).getData();
    }

    private String obtainUsername(String token) {
        Jwt jwt = this.convertJwt(token);

        return jwt.getClaimAsString(AuthenticationClaim.KEY_USERNAME);
    }

    private Long obtainUserId(String token) {
        Jwt jwt = this.convertJwt(token);

        return Long.parseLong(jwt.getClaimAsString(AuthenticationClaim.KEY_USERID));
    }

    private Jwt convertJwt(String accessToken) {
        try {
            return jwtDecoder.decode(accessToken);
        } catch (JwtException e) {
            throw new AuthorizationException("认证异常，token不支持");
        }
    }

    @Autowired
    public void setObjectMapper(ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
    }

    @Autowired(required = false)
    public void setOidcUserResolver(OidcUserResolver oidcUserResolver) {
        this.oidcUserResolver = oidcUserResolver;
    }

    @Autowired(required = false)
    public void setCasUserResolver(CasUserResolver casUserResolver) {
        this.casUserResolver = casUserResolver;
    }

    @Autowired
    public void setJwtDecoder(JwtDecoder jwtDecoder) {
        this.jwtDecoder = jwtDecoder;
    }

    @Autowired(required = false)
    public void setAuthorizeSettingCustomizer(AuthorizeSettingCustomizer authorizeSettingCustomizer) {
        this.authorizeSettingCustomizer = authorizeSettingCustomizer;
    }
}
