package com.elitescloud.boot.auth.provider.provider;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ArrayUtil;
import com.elitescloud.boot.auth.client.common.OAuth2ClientConstant;
import com.elitescloud.boot.auth.client.token.AbstractCustomAuthenticationToken;
import com.elitescloud.boot.auth.provider.common.AuthorizationConstant;
import com.elitescloud.boot.auth.provider.config.properties.AuthorizationProviderProperties;
import com.elitescloud.boot.constant.AuthenticationClaim;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.Authentication;
import org.springframework.security.oauth2.jwt.JwtClaimsSet;
import org.springframework.security.oauth2.server.authorization.token.JwtEncodingContext;
import org.springframework.security.oauth2.server.authorization.token.OAuth2TokenCustomizer;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import java.util.Arrays;
import java.util.Collection;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2025/7/9 周三
 */
public class CloudtTokenCustomizer implements OAuth2TokenCustomizer<JwtEncodingContext> {
    private static final Logger logger = LoggerFactory.getLogger(CloudtTokenCustomizer.class);

    private final AuthorizationProviderProperties authorizationProviderProperties;

    public CloudtTokenCustomizer(AuthorizationProviderProperties authorizationProviderProperties) {
        this.authorizationProviderProperties = authorizationProviderProperties;
    }

    @Override
    public void customize(JwtEncodingContext context) {
        Authentication authentication = context.getPrincipal();
        Object principal = authentication.getPrincipal();
        JwtClaimsSet.Builder claims = context.getClaims();

        // 客户端信息
        var client = context.getRegisteredClient();
        if (client != null) {
            claims.claim(AuthenticationClaim.KEY_CLIENT_ID, client.getClientId());

            String tenants = client.getClientSettings().getSetting(OAuth2ClientConstant.SETTING_TENANT);
            if (StringUtils.hasText(tenants)) {
                claims.claim(AuthenticationClaim.KEY_AUTHED_TENANTS, tenants);
            }
        } else {
            var clientId = (String) HttpServletUtil.currentRequest().getAttribute(AuthorizationConstant.REQUEST_ATTRIBUTE_CLIENT_ID);
            if (StringUtils.hasText(clientId)) {
                claims.claim(AuthenticationClaim.KEY_CLIENT_ID, clientId);
            }
        }

        // 用户信息
        if (principal instanceof GeneralUserDetails) {
            GeneralUserDetails userDetails = (GeneralUserDetails) principal;

            claims.claim(AuthenticationClaim.KEY_USERNAME, userDetails.getUsername());
            claims.claim(AuthenticationClaim.KEY_USERID, userDetails.getUser().getId());
            if (userDetails.getTenant() != null) {
                claims.claim(AuthenticationClaim.KEY_TENANT_ID, userDetails.getTenant().getId());
            }
            if (userDetails.getUser().getCasUserId() != null) {
                claims.claim(AuthenticationClaim.KEY_CAS_USERID, userDetails.getUser().getCasUserId());
            }

            claims.claim(AuthenticationClaim.KEY_PRINCIPAL_TYPE, AuthenticationClaim.VALUE_PRINCIPAL_USER);
        } else {
            // 客户端信息
            claims.claim(AuthenticationClaim.KEY_PRINCIPAL_TYPE, AuthenticationClaim.VALUE_PRINCIPAL_CLIENT);
        }

        // 登录信息
        if (authentication instanceof AbstractCustomAuthenticationToken) {
            var cusAuth = (AbstractCustomAuthenticationToken) authentication;
            claims.claim(AuthenticationClaim.KEY_LOGIN_TYPE, cusAuth.loginType().getType());
            claims.claim(AuthenticationClaim.KEY_TERMINAL, cusAuth.getTerminal() == null ? "" : cusAuth.getTerminal().name());
        }

        // 自定义表单负载
        this.customizeForLoginForm(claims);
    }

    private void customizeForLoginForm(JwtClaimsSet.Builder claims) {
        if (CollUtil.isEmpty(authorizationProviderProperties.getLogin().getTokenPayloads())) {
            return;
        }

        var request = HttpServletUtil.currentRequest();
        if (request == null) {
            logger.warn("当前请求为空");
            return;
        }

        for (String paramName : authorizationProviderProperties.getLogin().getTokenPayloads()) {
            var values = obtainParams(request, paramName);
            if (ArrayUtil.isEmpty(values)) {
                request.getAttribute(paramName);
                continue;
            }

            values = Arrays.stream(values).filter(CharSequenceUtil::isNotBlank).toArray(String[]::new);
            if (values.length == 0) {
                continue;
            }
            if (values.length == 1) {
                claims.claim(paramName, values[0]);
            } else {
                claims.claim(paramName, values);
            }
        }
    }

    private String[] obtainParams(HttpServletRequest request, String paramName) {
        var values = request.getParameterValues(paramName);
        if (ArrayUtil.isNotEmpty(values)) {
            return values;
        }

        Object attribute = request.getAttribute(paramName);
        if (attribute == null) {
            return null;
        }

        if (attribute instanceof String) {
            return new String[]{(String) attribute};
        }

        if (attribute.getClass().isArray()) {
            return Arrays.stream((Object[]) attribute).map(Object::toString).toArray(String[]::new);
        }

        if (attribute instanceof Collection) {
            return ((Collection<?>) attribute).stream().map(Object::toString).toArray(String[]::new);
        }

        return null;
    }

}
