package com.elitescloud.boot.auth.provider.provider.alipay;

import com.alipay.api.AlipayClient;
import com.alipay.api.AlipayConfig;
import com.alipay.api.AlipayConstants;
import com.alipay.api.DefaultAlipayClient;
import com.alipay.api.internal.util.AlipayEncrypt;
import com.alipay.api.internal.util.AlipaySignature;
import com.alipay.api.request.AlipaySystemOauthTokenRequest;
import com.alipay.api.response.AlipaySystemOauthTokenResponse;
import com.elitescloud.boot.auth.provider.common.param.AlipayApp;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

/**
 * 支付宝工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2025/1/24
 */
public class AlipayTool {
    private static final Charset CHARSET = StandardCharsets.UTF_8;
    private static final String SERVER_URL = "https://openapi.alipay.com/gateway.do";

    /**
     * 换取授权令牌
     *
     * @param app  应用信息
     * @param code 授权码
     * @return 令牌信息
     * @throws Exception 异常信息
     */
    public static AlipaySystemOauthTokenResponse systemOAuthToken(@NotNull AlipayApp app, @NotBlank String code) throws Exception {
        // 初始化SDK
        AlipayClient alipayClient = new DefaultAlipayClient(convertConfig(app));

        // 构造请求参数以调用接口
        AlipaySystemOauthTokenRequest request = new AlipaySystemOauthTokenRequest();

        // 设置授权码
        request.setCode(code);

        // 设置授权方式
        request.setGrantType("authorization_code");

        return alipayClient.execute(request);
    }

    private static AlipayConfig convertConfig(AlipayApp app) {
        AlipayConfig config = new AlipayConfig();
        config.setServerUrl(SERVER_URL);
        config.setAppId(app.getAppId());
        config.setFormat(AlipayConstants.FORMAT_JSON);
        config.setCharset(CHARSET.name());
        config.setSignType(AlipayConstants.SIGN_TYPE_RSA2);
        config.setPrivateKey(app.getPrivateKey());
        config.setAlipayPublicKey(app.getPublicKeyAlipay());
        return config;
    }

    public static class RSA2 {
        private static final String ASYMMETRIC_TYPE = "RSA";
        private static final String SIGNATURE_ALGORITHM = "SHA256WithRSA";

        /**
         * RSA2最大加密明文大小(2048/8-11=244)
         */
        private static final int MAX_ENCRYPT_BLOCK_SIZE = 244;
        /**
         * RSA2最大解密密文大小(2048/8=256)
         */
        private static final int MAX_DECRYPT_BLOCK_SIZE = 256;

        public static String sign(@NotBlank String content, @NotBlank String privateKeyStr) throws Exception {
            return AlipaySignature.rsa256Sign(content, privateKeyStr, CHARSET.name());
        }

        /**
         * 验证签名
         *
         * @param content      待签名内容
         * @param sign         签名
         * @param publicKeyStr 公钥
         * @return 验签结果
         */
        public static boolean verifySign(@NotBlank String content, @NotBlank String sign, @NotBlank String publicKeyStr) throws Exception {
            return AlipaySignature.rsa256CheckContent(content, sign, publicKeyStr, CHARSET.name());
        }

        /**
         * 加密
         *
         * @param content      待加密内容
         * @param publicKeyStr 公钥
         * @return 加密结果
         */
        public static String encrypt(@NotBlank String content, @NotBlank String publicKeyStr) throws Exception {
            return AlipaySignature.encrypt(content, publicKeyStr, CHARSET.name(), AlipayConstants.SIGN_TYPE_RSA2);
        }

        /**
         * 解密
         *
         * @param cipherText    密文
         * @param privateKeyStr 私钥
         * @return 解密结果
         */
        public static String decrypt(@NotBlank String cipherText, @NotBlank String privateKeyStr) throws Exception {
            return AlipaySignature.decrypt(cipherText, privateKeyStr, CHARSET.name(), AlipayConstants.SIGN_TYPE_RSA2);
        }
    }

    public static class RSA {
        private static final String ASYMMETRIC_TYPE = "RSA";
        private static final String SIGNATURE_ALGORITHM = "SHA1WithRSA";

        /**
         * RSA最大加密明文大小(1024/8-11=117)
         */
        private static final int MAX_ENCRYPT_BLOCK_SIZE = 117;

        /**
         * RSA最大解密密文大小(1024/8=128)
         */
        private static final int MAX_DECRYPT_BLOCK_SIZE = 128;

        /**
         * 签名
         *
         * @param content       待签名内容
         * @param privateKeyStr 私钥
         * @return 签名
         */
        public static String sign(@NotBlank String content, @NotBlank String privateKeyStr) throws Exception {
            return AlipaySignature.rsaSign(content, privateKeyStr, CHARSET.name());
        }

        /**
         * 验证签名
         *
         * @param content      待签名内容
         * @param sign         签名
         * @param publicKeyStr 公钥
         * @return 验签结果
         */
        public static boolean verifySign(@NotBlank String content, @NotBlank String sign, @NotBlank String publicKeyStr) throws Exception {
            return AlipaySignature.rsaCheckContent(content, sign, publicKeyStr, CHARSET.name());
        }

        /**
         * 加密
         *
         * @param content      待加密内容
         * @param publicKeyStr 公钥
         * @return 加密结果
         */
        public static String encrypt(@NotBlank String content, @NotBlank String publicKeyStr) throws Exception {
            return AlipaySignature.encrypt(content, publicKeyStr, CHARSET.name(), AlipayConstants.SIGN_TYPE_RSA);
        }

        /**
         * 解密
         *
         * @param cipherText    密文
         * @param privateKeyStr 私钥
         * @return 解密结果
         */
        public static String decrypt(@NotBlank String cipherText, @NotBlank String privateKeyStr) throws Exception {
            return AlipaySignature.decrypt(cipherText, privateKeyStr, CHARSET.name(), AlipayConstants.SIGN_TYPE_RSA);
        }
    }

    public static class AES {
        private static final String PADDING_ALGORITHM = "AES/CBC/PKCS5Padding";
        private static final String ALGORITHM = AlipayConstants.ENCRYPT_TYPE_AES;

        /**
         * AES 加密
         *
         * @param encryptKey AES 密钥
         * @param content    待加密文案
         * @return 加密后内容
         * @throws Exception 加密异常
         * @see <a href = 'https://opendocs.alipay.com/common/02mse3#%E8%87%AA%E8%A1%8C%E5%AE%9E%E7%8E%B0%20AES%20%E5%AF%86%E9%92%A5%E5%8A%A0%E8%A7%A3%E5%AF%86'>官方文档</a>
         */

        public static String encrypt(@NotBlank String encryptKey, @NotBlank String content) throws Exception {
            return AlipayEncrypt.encryptContent(content, ALGORITHM, encryptKey, CHARSET.name());
        }

        /**
         * @param content    密文
         * @param encryptKey aes密钥
         * @return 原文
         */
        public static String decrypt(@NotBlank String encryptKey, String content) throws Exception {
            return AlipayEncrypt.decryptContent(content, ALGORITHM, encryptKey, CHARSET.name());
        }
    }
}
