package com.elitescloud.boot.auth.provider.provider.wechat;

import com.elitescloud.boot.auth.provider.provider.wechat.param.*;
import com.elitescloud.boot.util.RestTemplateFactory;
import com.elitescloud.boot.util.RestTemplateHelper;
import com.fasterxml.jackson.core.type.TypeReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.lang.NonNull;

import javax.validation.constraints.NotBlank;
import java.util.Map;

/**
 * 微信工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/20
 */
public class WechatTool {
    private static final Logger logger = LoggerFactory.getLogger(WechatTool.class);
    private final RestTemplateHelper restTemplateHelper = RestTemplateHelper.instance(RestTemplateFactory.instance());

    private WechatTool() {
    }

    /**
     * 获取微信工具类
     *
     * @return
     */
    public static WechatTool getInstance() {
        return new WechatTool();
    }

    /**
     * 获取accessToken
     * <p>
     * 接口文档<a href="https://developers.weixin.qq.com/doc/offiaccount/Basic_Information/Get_access_token.html">地址</a>
     *
     * @param appId     应用ID
     * @param appSecret 应用密码
     * @return token
     */
    public WechatToken getAccessToken(@NotBlank String appId, @NotBlank String appSecret) {
        String url = "https://api.weixin.qq.com/cgi-bin/token?grant_type=client_credential&appid=APPID&secret=APPSECRET";

        url = url.replace("APPID", appId)
                .replace("APPSECRET", appSecret);

        WechatToken wechatToken = restTemplateHelper.exchangeSafely(url, HttpMethod.GET, null, new TypeReference<>() {
        });
        if (wechatToken.isSuccess()) {
            return wechatToken;
        }
        logger.error("获取微信accessToken失败, {}, {}", wechatToken.getErrCode(), wechatToken.getErrMsg());
        return null;
    }

    /**
     * 获取手机号（小程序）
     * <p>
     * 接口文档<a href="https://developers.weixin.qq.com/miniprogram/dev/OpenApiDoc/user-info/phone-number/getPhoneNumber.html">地址</a>
     *
     * @param accessToken token
     * @param code        授权码
     * @return token
     */
    public WechatPhoneInfo getPhoneNumber(@NotBlank String accessToken, @NotBlank String code) {
        String url = "https://api.weixin.qq.com/wxa/business/getuserphonenumber?access_token=ACCESS_TOKEN";

        url = url.replace("ACCESS_TOKEN", accessToken);

        WechatPhoneInfo phoneInfo = restTemplateHelper.exchangeSafely(url, HttpMethod.POST, new HttpEntity<>(Map.of("code", code)), new TypeReference<>() {
        });
        if (phoneInfo.isSuccess()) {
            return phoneInfo;
        }
        logger.error("获取小程序手机号失败, {}, {}", phoneInfo.getErrCode(), phoneInfo.getErrMsg());
        return null;
    }

    /**
     * 获取OpenId（小程序）
     * <p>
     * 接口文档<a href="https://developers.weixin.qq.com/miniprogram/dev/OpenApiDoc/user-login/code2Session.html">地址</a>
     *
     * @param appId  token
     * @param secret
     * @param jsCode
     * @return token
     */
    public WechatCode2Session jscode2Session(@NonNull String appId, @NonNull String secret, @NonNull String jsCode) {
        String url = "https://api.weixin.qq.com/sns/jscode2session?appid=APPID&secret=SECRET&js_code=JSCODE&grant_type=authorization_code";

        url = url.replace("APPID", appId).replace("SECRET", secret).replace("JSCODE", jsCode);

        WechatCode2Session sessionInfo = restTemplateHelper.exchangeSafely(url, HttpMethod.GET, null, new TypeReference<>() {
        });
        if (sessionInfo.isSuccess()) {
            return sessionInfo;
        }
        logger.error("获取小程序OpenId失败, {}, {}", sessionInfo.getErrCode(), sessionInfo.getErrMsg());
        return null;
    }


    /**
     * OAuth2 code换 accessToken
     * <p>
     * 服务号
     *
     * @param appId
     * @param secret
     * @param code
     * @return
     */
    public WechatAccessToken oauth2AccessToken(@NonNull String appId, @NonNull String secret, @NonNull String code) {
        String url = "https://api.weixin.qq.com/sns/oauth2/access_token?appid=APPID&secret=SECRET&code=CODE&grant_type=authorization_code";

        url = url.replace("APPID", appId)
                .replace("SECRET", secret)
                .replace("CODE", code)
        ;

        WechatAccessToken accessToken = restTemplateHelper.exchangeSafely(url, HttpMethod.GET, null, new TypeReference<>() {
        });
        if (accessToken.isSuccess()) {
            return accessToken;
        }
        logger.error("获取微信AccessToken失败, {}, {}", accessToken.getErrCode(), accessToken.getErrMsg());
        return null;
    }

    /**
     * 获取微信用户信息
     *
     * @param accessToken
     * @param openId
     * @return
     */
    public WechatUserInfo snsUserInfo(String accessToken, String openId) {
        String url = "https://api.weixin.qq.com/sns/userinfo?access_token=ACCESS_TOKEN&openid=OPENID&lang=zh_CN";

        url = url.replace("ACCESS_TOKEN", accessToken)
                .replace("OPENID", openId)
        ;

        WechatUserInfo userInfo = restTemplateHelper.exchangeSafely(url, HttpMethod.GET, null, new TypeReference<>() {
        });
        if (userInfo.isSuccess()) {
            return userInfo;
        }
        logger.error("获取微信用户信息失败, {}, {}", userInfo.getErrCode(), userInfo.getErrMsg());
        return null;
    }
}
