package com.elitescloud.boot.auth.provider.provider.wecom;

import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.auth.provider.provider.wecom.param.AccessToken;
import com.elitescloud.boot.auth.provider.provider.wecom.param.login.Code2UserInfoResult;
import com.elitescloud.boot.auth.provider.provider.wecom.param.login.OpenId2UserIdResult;
import com.elitescloud.boot.auth.provider.provider.wecom.param.login.Ticket2UserDetailResult;
import com.elitescloud.boot.auth.provider.provider.wecom.param.login.UserId2OpenIdResult;
import com.elitescloud.boot.util.RestTemplateFactory;
import com.elitescloud.boot.util.RestTemplateHelper;
import com.fasterxml.jackson.core.type.TypeReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;

import javax.validation.constraints.NotBlank;
import java.util.Map;

/**
 * 企业微信工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2024/8/16
 */
public class WeComTool {
    private static final Logger logger = LoggerFactory.getLogger(WeComTool.class);
    private static final RestTemplateHelper restTemplateHelper = RestTemplateHelper.instance(RestTemplateFactory.instance());

    /**
     * 获取Token
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/91039'>API文档</a>
     *
     * @param corpid     企业ID
     * @param corpsecret 应用的凭证秘钥
     * @return token信息
     */
    public static AccessToken getToken(@NotBlank String corpid, @NotBlank String corpsecret) {
        Assert.notBlank(corpid, "企业ID为空");
        Assert.notBlank(corpsecret, "应用的凭证秘钥为空");

        if (logger.isDebugEnabled()) {
            logger.debug("创建token：{}, {}", corpid, corpsecret);
        }

        String url = "https://qyapi.weixin.qq.com/cgi-bin/gettoken?corpid=" + corpid + "&corpsecret=" + corpsecret;
        return restTemplateHelper.exchangeSafely(url, HttpMethod.GET, null, new TypeReference<>() {
        });
    }

    /**
     * userid转openid
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/90202'>API文档</a>
     *
     * @param accessToken token
     * @param userId      成员ID
     * @return 删除结果
     */
    public static UserId2OpenIdResult userId2OpenId(@NotBlank String accessToken, @NotBlank String userId) {
        Assert.notBlank(accessToken, "AccessToken为空");
        Assert.notBlank(userId, "成员ID为空");

        if (logger.isDebugEnabled()) {
            logger.debug("userid转openid：{}", userId);
        }

        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/convert_to_openid?access_token=" + accessToken;
        return restTemplateHelper.exchangeSafely(url, HttpMethod.POST, new HttpEntity<>(Map.of("userid", userId)), new TypeReference<>() {
        });
    }

    /**
     * openid转userid
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/90202'>API文档</a>
     *
     * @param accessToken token
     * @param openId      成员的OpenId
     * @return 删除结果
     */
    public static OpenId2UserIdResult openId2UserId(@NotBlank String accessToken, @NotBlank String openId) {
        Assert.notBlank(accessToken, "AccessToken为空");
        Assert.notBlank(openId, "OpenId为空");

        if (logger.isDebugEnabled()) {
            logger.debug("openid转userid：{}", openId);
        }

        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/convert_to_userid?access_token=" + accessToken;
        return restTemplateHelper.exchangeSafely(url, HttpMethod.POST, new HttpEntity<>(Map.of("openid", openId)), new TypeReference<>() {
        });
    }

    /**
     * 获取用户登录身份
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/98176'>API文档</a>
     *
     * @param accessToken token
     * @param code      授权码
     * @return 身份信息结果
     */
    public static Code2UserInfoResult code2UserInfo(@NotBlank String accessToken, @NotBlank String code) {
        Assert.notBlank(accessToken, "AccessToken为空");
        Assert.notBlank(code, "授权码为空");

        if (logger.isDebugEnabled()) {
            logger.debug("code转用户身份：{}", code);
        }

        String url = "https://qyapi.weixin.qq.com/cgi-bin/auth/getuserinfo?access_token=" + accessToken + "&code=" + code;
        return restTemplateHelper.exchangeSafely(url, HttpMethod.GET, null, new TypeReference<>() {
        });
    }

    /**
     * 获取用户登录身份
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/98176'>API文档</a>
     *
     * @param accessToken token
     * @param ticket      票据
     * @return 身份信息结果
     */
    public static Ticket2UserDetailResult ticket2UserDetail(@NotBlank String accessToken, @NotBlank String ticket) {
        Assert.notBlank(accessToken, "AccessToken为空");
        Assert.notBlank(ticket, "票据为空");

        if (logger.isDebugEnabled()) {
            logger.debug("ticket转用户身份：{}", ticket);
        }

        String url = "https://qyapi.weixin.qq.com/cgi-bin/auth/getuserdetail?access_token=" + accessToken;
        return restTemplateHelper.exchangeSafely(url, HttpMethod.POST, new HttpEntity<>(Map.of("user_ticket", ticket)), new TypeReference<>() {
        });
    }
}
