package com.elitescloud.boot.auth.provider.provider.wecom;

import com.elitescloud.boot.auth.client.tool.RedisHelper;
import com.elitescloud.boot.auth.provider.common.WecomAppProvider;
import com.elitescloud.boot.auth.provider.common.param.WecomApp;
import com.elitescloud.boot.auth.provider.provider.wecom.param.BaseWecomResult;
import com.elitescloud.boot.auth.provider.provider.wecom.param.login.Code2UserInfoResult;
import com.elitescloud.boot.auth.provider.provider.wecom.param.login.Ticket2UserDetailResult;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.JSONUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.function.Function;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/9/23
 */
public class WecomTemplate {
    private static final Logger logger = LoggerFactory.getLogger(WecomTemplate.class);
    private static final String INNER_APP_CONCAT = "concat";

    private final WecomAppProvider appProvider;
    private final RedisHelper redisHelper;

    public WecomTemplate(WecomAppProvider appProvider, RedisHelper redisHelper) {
        this.appProvider = appProvider;
        this.redisHelper = redisHelper;
    }

    /**
     * 获取用户详细信息
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/98176'>API文档</a>
     *
     * @param corpId  企业ID
     * @param agentId 应用ID
     * @param code    授权码
     * @return 用户信息
     */
    public Code2UserInfoResult code2UserInfo(@NotBlank String corpId, @NotBlank String agentId, @NotBlank String code) {
        Assert.hasText(corpId, "企业ID为空");
        Assert.hasText(agentId, "应用ID为空");
        Assert.hasText(code, "授权码为空");

        return this.executeWithToken(corpId, agentId, token -> WeComTool.code2UserInfo(token, code));
    }

    /**
     * 获取用户详细信息
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/98176'>API文档</a>
     *
     * @param corpId  企业ID
     * @param agentId 应用ID
     * @param ticket  票据
     * @return 用户信息
     */
    public Ticket2UserDetailResult getUserByTicket(@NotBlank String corpId, @NotBlank String agentId, @NotBlank String ticket) {
        Assert.hasText(corpId, "企业ID为空");
        Assert.hasText(agentId, "应用ID为空");
        Assert.hasText(ticket, "票据为空");

        return this.executeWithToken(corpId, agentId, token -> WeComTool.ticket2UserDetail(token, ticket));
    }

    private <T extends BaseWecomResult> T executeWithToken(@NotBlank String corpId, @NotBlank String agentId, @NotNull Function<String, T> supplier) {
        // 获取token
        String accessToken = getAccessToken(corpId, agentId);
        T result = supplier.apply(accessToken);
        if (result.isSuccess()) {
            return result;
        }

        // 刷新token重试
        clearAccessToken(corpId, agentId);
        result = supplier.apply(accessToken);
        if (!result.isSuccess()) {
            logger.error("调用企业微信接口失败：{}", result);
        }

        return result;
    }

    private String getAccessToken(@NotBlank String corpId, @NotBlank String agentId) {
        // 先从缓存获取
        String key = "wecom:accesstoken:" + corpId + ":" + agentId;
        String accessToken = (String) redisHelper.execute(redisUtils -> redisUtils.get(key));
        if (StringUtils.hasText(accessToken)) {
            return accessToken;
        }

        // 调用微信接口生成token
        Assert.notNull(appProvider, "未获取到有效的企业微信配置");
        WecomApp app = null;
        if (INNER_APP_CONCAT.equals(agentId)) {
            // 通讯录
            app = appProvider.getContact(corpId);
        } else {
            app = appProvider.getApp(corpId, agentId);
        }
        Assert.notNull(app, agentId + "应用未配置");
        String appSecret = app.getCorpsecret();
        Assert.hasText(appSecret, "秘钥为空：" + agentId);
        var tokenResult = WeComTool.getToken(corpId, app.getCorpsecret());
        if (tokenResult == null || !tokenResult.isSuccess()) {
            logger.error("企业微信授权异常：{}, {}, {}", corpId, app.getCorpsecret(), JSONUtil.toJsonString(tokenResult));
            throw new BusinessException("企业微信授权异常，请稍后再试");
        }

        // 将token缓存
        redisHelper.execute(redisUtils -> redisUtils.set(key, tokenResult.getAccess_token(), tokenResult.getExpires_in() - 60));
        return tokenResult.getAccess_token();
    }

    private void clearAccessToken(String corpId, String agentId) {
        String key = "wecom:accesstoken:" + corpId + ":" + agentId;
        redisHelper.execute(redisUtils -> {
            redisUtils.del(key);
            return null;
        });
    }
}
