package com.elitescloud.boot.auth.provider.security.grant;

import com.elitescloud.boot.auth.client.token.AbstractCustomAuthenticationToken;
import com.elitescloud.boot.auth.provider.CustomAuthenticationProvider;
import com.elitescloud.boot.auth.provider.config.properties.AuthorizationProviderProperties;
import com.elitescloud.boot.auth.provider.provider.user.UserDetailManager;
import com.elitescloud.boot.auth.provider.security.AuthenticationCheckService;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.lang.NonNull;
import org.springframework.security.authentication.*;
import org.springframework.security.authentication.dao.AbstractUserDetailsAuthenticationProvider;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.mapping.GrantedAuthoritiesMapper;
import org.springframework.security.core.authority.mapping.NullAuthoritiesMapper;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsChecker;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.crypto.password.PasswordEncoder;

import java.util.Collection;

/**
 * 自定义的身份认证的父类.
 * <p>
 * 一般用于登录
 *
 * @author Kaiser（wang shao）
 * @date 2022/01/01
 */
@Log4j2
public abstract class AbstractCustomAuthenticationProvider<T extends AbstractCustomAuthenticationToken<T>> implements CustomAuthenticationProvider {

    /**
     * 是否将authentication中的principal转为字符串
     */
    private boolean forcePrincipalAsString = false;

    /**
     * 是否隐藏 用户未找到 的异常
     */
    private boolean hideUserNotFoundExceptions = true;

    /**
     * 用户前置检查
     */
    private UserDetailsChecker preAuthenticationChecks = new DefaultPreAuthenticationChecks();

    /**
     * 用户后置检查
     */
    private UserDetailsChecker postAuthenticationChecks = new DefaultPostAuthenticationChecks();

    private GrantedAuthoritiesMapper authoritiesMapper = new NullAuthoritiesMapper();

    protected PasswordEncoder passwordEncoder;
    protected UserDetailManager userDetailManager;
    protected AuthorizationProviderProperties configProperties;
    protected CredentialCheckable credentialCheckable;
    private ObjectProvider<AuthenticationCheckService> authenticationCheckServiceObjectProvider;

    /**
     * 根据令牌获取用户信息
     *
     * @param authentication 认证令牌
     * @return 用户信息
     * @throws AuthenticationException 认证异常
     */
    @NonNull
    protected abstract GeneralUserDetails retrieveUser(T authentication) throws AuthenticationException;

    /**
     * 扩展信息检查
     *
     * @param userDetails    用户信息
     * @param authentication 认证令牌
     * @throws AuthenticationException 认证异常
     */
    protected void additionalAuthenticationChecks(GeneralUserDetails userDetails, T authentication) throws AuthenticationException {
        // 检查默认的扩展信息
        customAdditionalAuthenticationChecks(userDetails, authentication);
    }

    @Override
    public Authentication authenticate(Authentication authentication) throws AuthenticationException {
        @SuppressWarnings("unchecked")
        T authenticationToken = (T) authentication;

        // 先获取用户信息
        GeneralUserDetails user = null;
        try {
            user = retrieveUser(authenticationToken);
        } catch (UsernameNotFoundException ex) {
            if (!this.hideUserNotFoundExceptions) {
                throw ex;
            }
            log.info("用户不存在：{}", authentication.getPrincipal());
            throw new BadCredentialsException("账号或密码错误");
        }

        // 用户检查
        this.preAuthenticationChecks.check(user);
        additionalAuthenticationChecks(user, authenticationToken);
        this.postAuthenticationChecks.check(user);

        Object principalToReturn = user;
        if (this.forcePrincipalAsString) {
            principalToReturn = user.getUsername();
        }

        // 返回成功Authentication
        return createSuccessAuthentications(principalToReturn, authenticationToken, user);
    }

    @Override
    public boolean supports(Class<?> authentication) {
        return getAuthenticationTokenType().isAssignableFrom(authentication);
    }

    /**
     * 获取AuthenticationToken class
     *
     * @return class
     */
    public abstract Class<T> getAuthenticationTokenType();

    /**
     * 获取MixinAuthenticationToken class
     *
     * @return class
     */
    public abstract Class<?> getMixinAuthenticationTokenType();

    public boolean isHideUserNotFoundExceptions() {
        return hideUserNotFoundExceptions;
    }

    public void setHideUserNotFoundExceptions(boolean hideUserNotFoundExceptions) {
        this.hideUserNotFoundExceptions = hideUserNotFoundExceptions;
    }

    public UserDetailsChecker getPreAuthenticationChecks() {
        return preAuthenticationChecks;
    }

    public void setPreAuthenticationChecks(UserDetailsChecker preAuthenticationChecks) {
        this.preAuthenticationChecks = preAuthenticationChecks;
    }

    public UserDetailsChecker getPostAuthenticationChecks() {
        return postAuthenticationChecks;
    }

    public void setPostAuthenticationChecks(UserDetailsChecker postAuthenticationChecks) {
        this.postAuthenticationChecks = postAuthenticationChecks;
    }

    public boolean isForcePrincipalAsString() {
        return forcePrincipalAsString;
    }

    public void setForcePrincipalAsString(boolean forcePrincipalAsString) {
        this.forcePrincipalAsString = forcePrincipalAsString;
    }

    public void setAuthoritiesMapper(GrantedAuthoritiesMapper authoritiesMapper) {
        this.authoritiesMapper = authoritiesMapper;
    }

    @Autowired(required = false)
    public void setUserDetailManager(UserDetailManager userDetailManager) {
        this.userDetailManager = userDetailManager;
    }

    @Autowired
    public void setPasswordEncoder(PasswordEncoder passwordEncoder) {
        this.passwordEncoder = passwordEncoder;
    }

    @Autowired
    public void setConfigProperties(AuthorizationProviderProperties configProperties) {
        this.configProperties = configProperties;
    }

    @Autowired
    public void setAuthenticationCheckServiceObjectProvider(ObjectProvider<AuthenticationCheckService> authenticationCheckServiceObjectProvider) {
        this.authenticationCheckServiceObjectProvider = authenticationCheckServiceObjectProvider;
    }

    @Autowired
    public void setCredentialCheckable(CredentialCheckable credentialCheckable) {
        this.credentialCheckable = credentialCheckable;
    }

    private void customAdditionalAuthenticationChecks(GeneralUserDetails userDetails, T authentication) {
        for (AuthenticationCheckService authenticationCheckService : authenticationCheckServiceObjectProvider) {
            authenticationCheckService.additionalAuthenticationChecks(userDetails, authentication);
        }
    }

    private Authentication createSuccessAuthentications(Object principal, T authentication, GeneralUserDetails user) {
        // 创建授权过的Authentication
        T result = null;
        Collection<? extends GrantedAuthority> authorities = this.authoritiesMapper.mapAuthorities(user.getAuthorities());
        try {
            var tokenConstructor = getAuthenticationTokenType().getDeclaredConstructor(Object.class, Object.class, Collection.class);
            result = tokenConstructor.newInstance(principal, authentication.getCredentials(), authorities);
        } catch (Exception e) {
            throw new AuthenticationServiceException("创建Authentication Success Token失败", e);
        }

        result.setPrincipal(principal);
        result.setDetails(authentication.getDetails());
        result.setTerminal(authentication.getTerminal());

        user.setLoginAccount(authentication.getPrincipal() == null ? null : authentication.getPrincipal().toString());

        log.debug("Authenticated user");
        return result;
    }

    /**
     * 默认的前置用户检查{@link AbstractUserDetailsAuthenticationProvider.DefaultPreAuthenticationChecks}
     */
    private static class DefaultPreAuthenticationChecks implements UserDetailsChecker {

        @Override
        public void check(UserDetails user) {
            if (user == null) {
                throw new UsernameNotFoundException("账号不存在");
            }
        }

    }

    /**
     * 默认的后置用户检查{@link AbstractUserDetailsAuthenticationProvider.DefaultPostAuthenticationChecks}
     */
    private static class DefaultPostAuthenticationChecks implements UserDetailsChecker {

        @Override
        public void check(UserDetails user) {
            if (!user.isCredentialsNonExpired()) {
                log.debug("Failed to authenticate since user account credentials have expired");
                throw new CredentialsExpiredException("密码已过期");
            }
            if (!user.isAccountNonLocked()) {
                log.debug("Failed to authenticate since user account is locked");
                throw new LockedException("账号已锁定");
            }
            if (!user.isEnabled()) {
                log.debug("Failed to authenticate since user account is disabled");
                throw new DisabledException("账号已禁用");
            }
            if (!user.isAccountNonExpired()) {
                log.debug("Failed to authenticate since user account has expired");
                throw new AccountExpiredException("账号已过期");
            }
        }
    }
}
