package com.elitescloud.boot.auth.provider.security.grant.wechat_phone;

import com.elitescloud.boot.auth.client.common.LoginType;
import com.elitescloud.boot.auth.client.token.AbstractCustomAuthenticationToken;
import com.elitescloud.boot.auth.provider.common.LoginParameterNames;
import com.elitescloud.cloudt.common.constant.Terminal;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import org.springframework.lang.NonNull;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.util.MultiValueMap;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import java.util.Collection;

/**
 * 微信openid的身份认证令牌.
 * <p>
 * 基于微信手机号的认证方式登录，<a href="https://developers.weixin.qq.com/miniprogram/dev/framework/open-ability/getPhoneNumber.html">获取手机号</a>
 *
 * @author Kaiser（wang shao）
 * @date 2022/01/01
 */
public class WechatPhoneAuthenticationToken extends AbstractCustomAuthenticationToken<WechatPhoneAuthenticationToken> {
    private static final long serialVersionUID = 35697051927011389L;

    /**
     * 账号不存在时是否自动 注册
     */
    private boolean autoRegister = false;

    public WechatPhoneAuthenticationToken() {
        super(null, null);
    }

    public WechatPhoneAuthenticationToken(Object principal, Object credentials) {
        super(principal, credentials);
    }

    public WechatPhoneAuthenticationToken(Object principal, Object credentials, Collection<? extends GrantedAuthority> authorities) {
        super(principal, credentials, authorities);
    }

    public boolean isAutoRegister() {
        return autoRegister;
    }

    public void setAutoRegister(boolean autoRegister) {
        this.autoRegister = autoRegister;
    }

    @NonNull
    @Override
    public LoginType loginType() {
        return LoginType.WX_MOBILE;
    }

    @NonNull
    @Override
    public WechatPhoneAuthenticationToken convert(@NonNull HttpServletRequest request) {
        MultiValueMap<String, String> parameters = HttpServletUtil.getParameters(request);
        String terminalParam = parameters.getFirst(LoginParameterNames.TERMINAL);
        Terminal terminal = null;
        if (StringUtils.hasText(terminalParam)) {
            terminal = Terminal.valueOf(terminalParam);
        }

        WechatPhoneAuthenticationToken authenticationToken = new WechatPhoneAuthenticationToken();
        authenticationToken.setTerminal(terminal);
        authenticationToken.setPrincipal(parameters.getFirst(LoginParameterNames.WECHAT_APPID));
        authenticationToken.setCredentials(parameters.getFirst(LoginParameterNames.WECHAT_CODE));
        authenticationToken.setAutoRegister("true".equalsIgnoreCase(parameters.getFirst(LoginParameterNames.AUTO_REGISTER)));
        authenticationToken.setAuthenticated(false);

        // 身份类型
        authenticationToken.setIdentity(parameters.getFirst(LoginParameterNames.IDENTITY_TYPE));

        return authenticationToken;
    }
}
