package com.elitescloud.boot.auth.provider.sso.impl;

import com.elitescloud.boot.auth.config.AuthorizationSdkProperties;
import com.elitescloud.boot.auth.provider.provider.user.UserDetailManager;
import com.elitescloud.boot.auth.provider.security.generator.ticket.TicketGenerator;
import com.elitescloud.boot.auth.provider.sso.SsoLoginSupportProvider;
import com.elitescloud.boot.auth.sso.common.SsoAccountType;
import com.elitescloud.boot.auth.sso.model.UserInfoDTO;
import com.elitescloud.boot.auth.util.AuthorizationServerHelper;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import lombok.extern.log4j.Log4j2;
import org.springframework.util.StringUtils;

/**
 * 云梯架构的单点登录支撑.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/15
 */
@Log4j2
public class CloudtSsoLoginSupportProviderImpl implements SsoLoginSupportProvider {

    private final AuthorizationSdkProperties sdkProperties;
    private final TicketGenerator ticketGenerator;
    private final UserDetailManager userDetailManager;

    public CloudtSsoLoginSupportProviderImpl(AuthorizationSdkProperties sdkProperties, TicketGenerator ticketGenerator,
                                             UserDetailManager userDetailManager) {
        this.sdkProperties = sdkProperties;
        this.ticketGenerator = ticketGenerator;
        this.userDetailManager = userDetailManager;
    }

    /**
     * 生成ticket
     *
     * @param account
     * @param accountType
     * @return
     */
    public ApiResult<String> generateTicket(String account, String accountType) {
        SsoAccountType ssoAccountType = null;
        try {
            ssoAccountType = SsoAccountType.valueOf(accountType);
        } catch (IllegalArgumentException e) {
            return ApiResult.fail("不支持的账号类型：" + accountType);
        }
        SysUserDTO user = null;
        try {
            user = queryUser(account, ssoAccountType);
        } catch (Exception e) {
            log.info("未查询到用户：{}，{}，{}", accountType, account, e.getMessage());
        }

        // 生成ticket
        String ticket = null;
        if (user != null) {
            ticket = ticketGenerator.generate(user);
        }
        return ApiResult.ok(ticket);
    }

    /**
     * 根据ticket获取用户信息
     *
     * @param ticket
     * @return
     */
    public ApiResult<UserInfoDTO> getUserInfo(String ticket) {
        if (!StringUtils.hasText(ticket)) {
            return ApiResult.fail("ticket为空");
        }

        var user = ticketGenerator.validate(ticket);
        return ApiResult.ok(user);
    }

    private SysUserDTO queryUser(String account, SsoAccountType accountType) {
        switch (accountType) {
            case USER_ID:
                return userDetailManager.loadUserById(account).getUser();
            case USER_NAME:
                return userDetailManager.loadUserByUsername(account).getUser();
            case MOBILE:
                return userDetailManager.loadUserByMobile(account).getUser();
            case EMAIL:
                return userDetailManager.loadUserByEmail(account).getUser();
            default:
                return userDetailManager.loadUserByUsername(account).getUser();
        }
    }
}
