package com.elitescloud.boot.auth.provider.sso.impl;

import com.elitescloud.boot.auth.client.common.AuthorizationException;
import com.elitescloud.boot.auth.client.common.SecurityConstants;
import com.elitescloud.boot.auth.client.config.security.handler.DelegateAuthenticationCallable;
import com.elitescloud.boot.auth.client.config.security.resolver.BearerTokenResolver;
import com.elitescloud.boot.auth.client.config.security.resolver.impl.DefaultBearerTokenResolver;
import com.elitescloud.boot.auth.client.config.support.AuthenticationCache;
import com.elitescloud.boot.auth.client.tool.RedisHelper;
import com.elitescloud.boot.auth.config.AuthorizationSdkProperties;
import com.elitescloud.boot.auth.model.OAuthToken;
import com.elitescloud.boot.auth.provider.provider.user.UserDetailManager;
import com.elitescloud.boot.auth.provider.security.generator.token.TokenGenerator;
import com.elitescloud.boot.auth.sso.SsoProvider;
import com.elitescloud.boot.auth.sso.common.SsoAccountType;
import com.elitescloud.boot.auth.sso.common.TicketAuthentication;
import com.elitescloud.boot.auth.sso.model.UserInfoDTO;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.log4j.Log4j2;
import org.springframework.security.core.AuthenticationException;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Collections;

/**
 * 云梯架构的sso提供者.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/15
 */
@Log4j2
public class CloudtSsoProviderImpl implements SsoProvider {

    private final TokenGenerator tokenGenerator;
    private final AuthenticationCache authenticationCache;
    private final UserDetailManager userDetailManager;
    private final RedisHelper redisHelper;
    private BearerTokenResolver bearerTokenResolver = new DefaultBearerTokenResolver();
    private AuthorizationSdkProperties sdkProperties;
    private DelegateAuthenticationCallable authenticationCallable;

    public CloudtSsoProviderImpl(TokenGenerator tokenGenerator, AuthenticationCache authenticationCache,
                                 UserDetailManager userDetailManager, RedisHelper redisHelper) {
        this.tokenGenerator = tokenGenerator;
        this.authenticationCache = authenticationCache;
        this.userDetailManager = userDetailManager;
        this.redisHelper = redisHelper;
    }

    @Override
    public boolean isAuthenticated(HttpServletRequest request) {
        String token = bearerTokenResolver.resolve(request);
        return authenticationCache.getUserDetail(token) != null;
    }

    @Override
    public TicketAuthentication authentication(HttpServletRequest request, HttpServletResponse response, TicketAuthentication authentication) throws AuthenticationException {
        // 校验authentication
        TicketAuthentication authenticationToken = validateAuthentication(authentication);

        // 生成token
        OAuthToken authToken = tokenGenerator.generate(authenticationToken);
        authenticationToken.setToken(authToken);

        // 保存token与ticket关联
        try {
            redisHelper.execute(redisUtils -> {
                redisUtils.set(SecurityConstants.CACHE_PREFIX_SSO_TOKEN + authentication.getTicket(), authToken.getAccessToken());
                return null;
            });

            // 回调处理
            authenticationCallable.onLogin(request, response, authToken.getAccessToken(), authenticationToken);
        } catch (Exception e) {
            log.error("单点登录保存token与ticket关联失败", e);
            throw new IllegalStateException("生成token失败", e);
        }

        return authenticationToken;
    }

    @Override
    public void clearToken(HttpServletRequest request, HttpServletResponse response, String ticket) {
        // 获取token
        String token = null;
        try {
            token = (String) redisHelper.execute(redisUtils -> redisUtils.get(SecurityConstants.CACHE_PREFIX_SSO_TOKEN + ticket));
        } catch (Exception e) {
            log.error("单点登录获取token与ticket关联失败", e);
            throw new IllegalStateException("获取token失败", e);
        }

        if (StringUtils.hasText(token)) {
            // 删除关联
            try {
                redisHelper.execute(redisUtils -> {
                    redisUtils.del(SecurityConstants.CACHE_PREFIX_SSO_TOKEN + ticket);
                    return null;
                });
            } catch (Exception e) {
                log.error("单点登录获取token与ticket关联失败", e);
            }

            // 删除token
            try {
                authenticationCache.removeUserDetail(token);
            } catch (Exception e) {
                log.error("删除用户token失败", e);
            }
        }
    }

    public void setBearerTokenResolver(BearerTokenResolver bearerTokenResolver) {
        this.bearerTokenResolver = bearerTokenResolver;
    }

    public void setSdkProperties(AuthorizationSdkProperties sdkProperties) {
        this.sdkProperties = sdkProperties;
    }

    public void setAuthenticationCallable(DelegateAuthenticationCallable authenticationCallable) {
        this.authenticationCallable = authenticationCallable;
    }

    private TicketAuthentication validateAuthentication(TicketAuthentication authentication) {
        // 判断账号类型
        SsoAccountType accountType = null;
        if (sdkProperties != null) {
            accountType = sdkProperties.getSso().getAccountType();
        }
        if (accountType == null) {
            accountType = SsoAccountType.USER_NAME;
        }

        // 查询用户信息
        UserInfoDTO userInfoDTO = authentication.getUserInfoDTO();
        GeneralUserDetails userDetails = null;
        switch (accountType) {
            case USER_ID:
                userDetails = userDetailManager.loadUserById(userInfoDTO.getUserId());
                break;
            case USER_NAME:
                userDetails = userDetailManager.loadUserByUsername(userInfoDTO.getUsername());
                break;
            case MOBILE:
                userDetails = userDetailManager.loadUserByMobile(userInfoDTO.getMobile());
                break;
            case EMAIL:
                userDetails = userDetailManager.loadUserByEmail(userInfoDTO.getEmail());
                break;
        }

        if (userDetails == null) {
            throw new AuthorizationException("用户不存在");
        }

        TicketAuthentication authenticationToken = new TicketAuthentication(authentication.getTicket(), userInfoDTO, Collections.emptyList());
        authenticationToken.setPrincipal(userDetails);

        return authenticationToken;
    }
}
