package com.elitescloud.boot.auth.provider.sso2.support.convert;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ArrayUtil;
import com.elitescloud.boot.auth.client.common.AuthorizationException;
import com.elitescloud.boot.auth.provider.security.grant.InternalAuthenticationGranter;
import com.elitescloud.boot.auth.provider.sso2.common.SsoConvertProperty;
import com.elitescloud.boot.auth.provider.sso2.common.SsoTypeEnum;
import com.elitescloud.boot.auth.provider.sso2.support.convert.properties.LdapSsoConvertProperty;
import org.jetbrains.annotations.Nullable;
import org.springframework.boot.context.properties.PropertyMapper;
import org.springframework.ldap.core.LdapTemplate;
import org.springframework.ldap.core.support.LdapContextSource;
import org.springframework.ldap.filter.AndFilter;
import org.springframework.ldap.filter.EqualsFilter;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * LDAP类型的.
 *
 * @author Kaiser（wang shao）
 * @date 2025/6/8 周日
 */
public class LdapSsoAuthenticationConvert extends BasePlainSsoAuthenticationConvert {

    @Override
    public SsoTypeEnum supportType() {
        return SsoTypeEnum.LDAP;
    }

    @Override
    public <T extends SsoConvertProperty> Class<T> propertyType() {
        return (Class<T>) LdapSsoConvertProperty.class;
    }

    @Nullable
    @Override
    public <T extends SsoConvertProperty> InternalAuthenticationGranter.InternalAuthenticationToken convert(HttpServletRequest request, HttpServletResponse response, T properties) {
        LdapSsoConvertProperty props = (LdapSsoConvertProperty) properties;

        String value = getParam(request, props.getParamName(), props.getParamIn());
        if (CharSequenceUtil.isBlank(value)) {
            throw new IllegalArgumentException("参数为空:" + props.getParamName());
        }

        // 获取账户名密码
        String[] credentials = decodeBasicAuth(value);
        if (ArrayUtil.isEmpty(credentials) || CharSequenceUtil.isBlank(credentials[0])) {
            throw new IllegalArgumentException("授权账户为空");
        }

        // 校验账户密码
        boolean authenticated = authenticateByLdap(credentials[0], credentials[1], props);
        if (!authenticated) {
            throw new AuthorizationException("认证失败, 请确认账户密码一致");
        }

        return new InternalAuthenticationGranter.InternalAuthenticationToken(props.getIdType(), credentials[0]);
    }

    private String[] decodeBasicAuth(String basicAuth) {
        if (basicAuth.startsWith("Basic ") || basicAuth.startsWith("basic ")) {
            basicAuth = basicAuth.substring(6);
        }

        String credentialsString = new String(Base64.getDecoder().decode(basicAuth.getBytes(StandardCharsets.UTF_8)));
        String[] credentials = credentialsString.split(":");

        return credentials.length == 1 ? new String[] {URLDecoder.decode(credentials[0], StandardCharsets.UTF_8)} :
                new String[] {URLDecoder.decode(credentials[0], StandardCharsets.UTF_8), URLDecoder.decode(credentials[1], StandardCharsets.UTF_8)};
    }

    private boolean authenticateByLdap(String principal, String password, LdapSsoConvertProperty props) {
        // 通用条件
        Map<String, Object> attributes = new HashMap<>(8);
        if (CollUtil.isNotEmpty(props.getLoginAttributes())) {
            attributes.putAll(props.getLoginAttributes());
        }

        AndFilter filter = new AndFilter().and(new EqualsFilter(props.getLoginAttributeName(), principal));
        for (Map.Entry<String, Object> entry : attributes.entrySet()) {
            filter.and(entry.getValue() instanceof Integer ? new EqualsFilter(entry.getKey(), (Integer) entry.getValue()) : new EqualsFilter(entry.getKey(), entry.getValue().toString()));
        }

        try {
            LdapTemplate ldapTemplate = buildLdapTemplate(props);
            return ldapTemplate.authenticate(props.getBase(), filter.encode(), password);
        } catch (Exception e) {
            throw new AuthorizationException("LDAP认证失败," + e.getMessage(), e);
        }
    }

    private LdapTemplate buildLdapTemplate(LdapSsoConvertProperty props) {
        var contextSource = this.buildLdapContextSource(props);
        LdapSsoConvertProperty.Template template = props.getTemplate();
        PropertyMapper propertyMapper = PropertyMapper.get().alwaysApplyingWhenNonNull();
        LdapTemplate ldapTemplate = new LdapTemplate(contextSource);
        propertyMapper.from(template.isIgnorePartialResultException())
                .to(ldapTemplate::setIgnorePartialResultException);
        propertyMapper.from(template.isIgnoreNameNotFoundException()).to(ldapTemplate::setIgnoreNameNotFoundException);
        propertyMapper.from(template.isIgnoreSizeLimitExceededException())
                .to(ldapTemplate::setIgnoreSizeLimitExceededException);

        return ldapTemplate;
    }

    private LdapContextSource buildLdapContextSource(LdapSsoConvertProperty props) {
        Assert.notEmpty(props.getUrls(), "LDAP url未配置");

        LdapContextSource source = new LdapContextSource();
        PropertyMapper propertyMapper = PropertyMapper.get().alwaysApplyingWhenNonNull();
        propertyMapper.from(props.getUsername()).to(source::setUserDn);
        propertyMapper.from(props.getPassword()).to(source::setPassword);
        propertyMapper.from(props.getAnonymousReadOnly()).to(source::setAnonymousReadOnly);
        propertyMapper.from(props.getBase()).to(source::setBase);
        propertyMapper.from(props.getUrls()).to(source::setUrls);
        propertyMapper.from(props.getBaseEnvironment())
                .to((baseEnvironment) -> source.setBaseEnvironmentProperties(Collections.unmodifiableMap(baseEnvironment)));

        return source;
    }
}
