package com.elitescloud.boot.auth.provider.sso2.support.impl;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.auth.cas.provider.ClientTransferHelper;
import com.elitescloud.boot.auth.config.AuthorizationSdkProperties;
import com.elitescloud.boot.auth.provider.config.properties.Sso2Properties;
import com.elitescloud.boot.auth.provider.sso2.common.SsoUserInfoConvert;
import com.elitescloud.boot.auth.provider.sso2.common.TicketProvider;
import com.elitescloud.boot.auth.provider.sso2.support.SsoServerSupportProvider;
import com.elitescloud.boot.auth.sso.common.SdkSsoConstants;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.security.core.Authentication;
import org.springframework.security.oauth2.server.authorization.authentication.OAuth2ClientAuthenticationToken;
import org.springframework.security.oauth2.server.authorization.web.authentication.ClientSecretBasicAuthenticationConverter;
import org.springframework.security.oauth2.server.authorization.web.authentication.ClientSecretPostAuthenticationConverter;
import org.springframework.security.web.authentication.AuthenticationConverter;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.List;

/**
 * 默认的sso支持实现.
 *
 * @author Kaiser（wang shao）
 * @date 2024/2/20
 */
@Slf4j
public class DefaultSsoServerSupportProvider extends BaseSsoSupportProvider implements SsoServerSupportProvider {

    protected final Sso2Properties sso2Properties;
    protected final TicketProvider ticketProvider;
    private final AuthorizationSdkProperties sdkProperties;
    private final List<SsoUserInfoConvert<?>> userInfoConverts;

    private final ClientTransferHelper clientTransferHelper;
    private final List<AuthenticationConverter> authenticationConverters = List.of(new ClientSecretBasicAuthenticationConverter(),
            new ClientSecretPostAuthenticationConverter());

    public DefaultSsoServerSupportProvider(Sso2Properties sso2Properties, TicketProvider ticketProvider,
                                           List<SsoUserInfoConvert<?>> userInfoConverts, AuthorizationSdkProperties sdkProperties) {
        super(ticketProvider);
        this.sso2Properties = sso2Properties;
        this.ticketProvider = ticketProvider;
        this.userInfoConverts = userInfoConverts;
        this.sdkProperties = sdkProperties;
        this.clientTransferHelper = ClientTransferHelper.getInstance(sdkProperties.getAuthServer());
    }

    @Override
    public ApiResult<String> generateTicket(HttpServletRequest request, HttpServletResponse response) {
        // 生成ticket
        var ticket = ticketProvider.generateTicket(request, response);

        return ApiResult.ok(ticket);
    }

    @Override
    public ApiResult<Object> getUserDetail(HttpServletRequest request, HttpServletResponse response) {
        // 校验客户端
        var client = this.obtainClient(request);
        var validated = this.validateClient(request, client);
        var clientId = client == null ? null : client.getClientId();
        log.info("sso ticket2user：{}，{}", clientId, validated);
        if (!validated) {
            response.setStatus(HttpServletResponse.SC_UNAUTHORIZED);
            return ApiResult.fail(ApiCode.UNAUTHORIZED, "客户端验证失败");
        }

        // 获取ticket
        var ticket = this.obtainTicket(request);
        if (CharSequenceUtil.isBlank(ticket)) {
            return ApiResult.fail("票据ticket为空");
        }
        log.info("sso ticket：{}", ticket);

        // 获取用户信息
        var userDetails = this.retrieveUserDetails(request, ticket);
        if (userDetails == null) {
            return ApiResult.fail("票据ticket不存在或已过期，请重新登录");
        }
        var userInfo = this.convertSsoUserInfo(userDetails.getUser(), clientId);
        log.info("sso convert user：{}, {}", clientId, userInfo);

        return ApiResult.ok(userInfo);
    }

    @Override
    public ApiResult<String> ticketToToken(String ticket) {

        return null;
    }

    protected Object convertSsoUserInfo(SysUserDTO userDTO, String clientId) {
        if (userDTO == null || CharSequenceUtil.isBlank(clientId)) {
            return userDTO;
        }

        for (var userInfoConvert : userInfoConverts) {
            if (clientId.equals(userInfoConvert.supportClientId())) {
                return userInfoConvert.convertUser(userDTO);
            }
        }

        return userDTO;
    }

    protected boolean validateClient(HttpServletRequest request, ClientInfo client) {
        if (Boolean.FALSE.equals(sso2Properties.getServer().getValidateClient())) {
            // 不需要验证
            return true;
        }

        // 获取传递的客户端信息
        if (client == null) {
            log.info("sso验证客户端异常，未获取到客户端信息");
            return false;
        }

        var prop = sdkProperties.getCasClient().getOauth2Client();
        if (prop == null || CharSequenceUtil.isBlank(prop.getClientId())) {
            log.warn("未配置OAuth2客户端");
            return false;
        }

        // 判断本地是否已配置
        if (prop.getClientId().equals(client.getClientId())) {
            if (CharSequenceUtil.isBlank(prop.getClientSecret())) {
                return CharSequenceUtil.isBlank(client.getClientSecret());
            }
            return prop.getClientSecret().equals(client.getClientSecret());
        }

        // 调用统一身份认证去验证
        var validatedResult = clientTransferHelper.validateClient(client.getClientId(), client.getClientSecret());
        if (Boolean.TRUE.equals(validatedResult.getSuccess()) && Boolean.TRUE.equals(validatedResult.getData())) {
            return true;
        }
        log.info("校验客户端不通过：{}，{}，{}", client.getClientId(), client.getClientSecret(), validatedResult);

        return false;
    }

    protected GeneralUserDetails retrieveUserDetails(HttpServletRequest request, String ticket) {
        // 获取票据对应的token
        var token = (String) ticketProvider.exchangeTicket(ticket);
        if (CharSequenceUtil.isBlank(token)) {
            return null;
        }

        // 根据token获取用户
        var user = SecurityContextUtil.convertToken(token);

        user.getUser().setPassword(null);
        return user;
    }

    protected String obtainTicket(HttpServletRequest request) {
        // 优先从请求参数中获取ticket
        var ticket = request.getParameter(SdkSsoConstants.PARAM_SSO_TICKET);
        if (StringUtils.hasText(ticket)) {
            return ticket;
        }

        // 其次从请求头获取
        ticket = request.getHeader(SdkSsoConstants.HEADER_SSO_TICKET);
        return ticket;
    }

    protected ClientInfo obtainClient(HttpServletRequest request) {
        for (AuthenticationConverter authenticationConverter : authenticationConverters) {
            Authentication authenticationToken = null;
            try {
                authenticationToken = authenticationConverter.convert(request);
            } catch (Exception e) {
                log.warn("解析客户端异常：", e);
                continue;
            }
            if (authenticationToken == null) {
                continue;
            }
            if (authenticationToken instanceof OAuth2ClientAuthenticationToken) {
                var clientToken = ((OAuth2ClientAuthenticationToken) authenticationToken);
                return new ClientInfo((String) clientToken.getPrincipal(), (String) clientToken.getCredentials());
            }
            log.warn("解析认证客户端异常：{}", authenticationToken.getClass().getName());
        }

        return null;
    }

    protected static class ClientInfo {
        private final String clientId;
        private final String clientSecret;

        public ClientInfo(String clientId, String clientSecret) {
            this.clientId = clientId;
            this.clientSecret = clientSecret;
        }

        public String getClientId() {
            return clientId;
        }

        public String getClientSecret() {
            return clientSecret;
        }
    }
}
