package com.elitescloud.boot.auth.sso.impl;

import com.elitescloud.boot.auth.config.AuthorizationSdkProperties;
import com.elitescloud.boot.auth.sso.TicketResolver;
import com.elitescloud.boot.auth.sso.common.SdkSsoConstants;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * cookie中解析ticket.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/14
 */
public class CookieTicketResolver implements TicketResolver {
    private static final Logger logger = LoggerFactory.getLogger(CookieTicketResolver.class);

    private final AuthorizationSdkProperties sdkProperties;

    public CookieTicketResolver(AuthorizationSdkProperties sdkProperties) {
        this.sdkProperties = sdkProperties;
    }

    @Override
    public String obtain(HttpServletRequest request) {
        // 先从请求参数中获取
        String ticket = request.getParameter(SdkSsoConstants.PARAM_SSO_TICKET);
        if (StringUtils.hasText(ticket)) {
            return ticket;
        }

        // 再从请求头中获取
        ticket = request.getHeader(SdkSsoConstants.HEADER_SSO_TICKET);
        if (StringUtils.hasText(ticket)) {
            return ticket;
        }

        // 最后从cookie中获取
        Cookie[] cookies = request.getCookies();
        if (cookies == null || cookies.length == 0) {
            return null;
        }

        for (Cookie cookie : cookies) {
            if (SdkSsoConstants.COOKIE_SSO_TICKET.equals(cookie.getName())) {
                logger.info("获取到ticket：{}", cookie.getValue());
                return cookie.getValue();
            }
        }


        return null;
    }

    @Override
    public void save(HttpServletRequest request, HttpServletResponse response, String ticket) {
        AuthorizationSdkProperties.Sso sso = sdkProperties.getSso();
        writeToCookie(ticket, sso.getCookieDomains(), sso.getCookieHttpOnly(), sso.getCookieSecure(), response);
    }

    @Override
    public void clear(HttpServletRequest request, HttpServletResponse response) {
        clearCookie(sdkProperties.getSso().getCookieDomains(), response);
    }

    private void writeToCookie(String ticket, Set<String> domains, boolean httpOnly, boolean secure, HttpServletResponse response) {
        if (CollectionUtils.isEmpty(domains)) {
            logger.error("sso ticket写入cookie失败，域名未配置");
            return;
        }

        var cookies = domains.stream()
                .map(t -> {
                    Cookie cookie = new Cookie(SdkSsoConstants.COOKIE_SSO_TICKET, ticket);
                    cookie.setDomain(t);
                    cookie.setPath("/");
                    cookie.setHttpOnly(httpOnly);
                    cookie.setSecure(secure);

                    return cookie;
                }).collect(Collectors.toList());
        for (Cookie cookie : cookies) {
            response.addCookie(cookie);
        }
    }

    private void clearCookie(Set<String> domains, HttpServletResponse response) {
        if (CollectionUtils.isEmpty(domains)) {
            logger.error("sso ticket清理cookie失败，域名未配置");
            return;
        }

        var cookies = domains.stream()
                .map(t -> {
                    Cookie cookie = new Cookie(SdkSsoConstants.COOKIE_SSO_TICKET, null);
                    cookie.setDomain(t);
                    cookie.setPath("/");
                    cookie.setMaxAge(0);

                    return cookie;
                }).collect(Collectors.toList());
        for (Cookie cookie : cookies) {
            response.addCookie(cookie);
        }
    }
}
