package com.elitescloud.boot.auth.cas.provider;

import com.elitescloud.boot.auth.cas.task.ClientTokenHolder;
import com.elitescloud.boot.auth.common.AuthSdkConstant;
import com.elitescloud.boot.auth.model.UpdaterInfoDTO;
import com.elitescloud.boot.common.OpenApiException;
import com.elitescloud.boot.util.RestTemplateFactory;
import com.elitescloud.boot.util.RestTemplateHelper;
import com.fasterxml.jackson.core.type.TypeReference;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.*;
import org.springframework.http.client.ClientHttpRequestExecution;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.lang.NonNull;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.util.StringUtils;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.time.Duration;
import java.util.Arrays;
import java.util.List;
import java.util.StringJoiner;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2/14/2023
 */
public abstract class BaseTransferHelper {
    private static final Logger LOG = LoggerFactory.getLogger(BaseTransferHelper.class);

    private final RestTemplateHelper restTemplateHelper;
    private static final Cache<String, Object> EXCHANGE_CACHE = Caffeine.newBuilder()
            .maximumSize(1000)
            .expireAfterWrite(Duration.ofMinutes(1))
            .build();

    BaseTransferHelper(String authServer) {
        this.restTemplateHelper = RestTemplateHelper.instance(
                RestTemplateFactory.dynamicInstance(builder -> builder.rootUri(authServer)
                        .additionalInterceptors(new AuthorizationInterceptor()), AuthSdkConstant.serverName)
        );
    }

    <T> T nullToDefault(T obj, T defaultObj) {
        return obj == null ? defaultObj : obj;
    }

    <T> T remoteExchange(String url, HttpMethod httpMethod, HttpEntity<?> httpEntity, TypeReference<T> responseType,
                         Object... param) {
        try {
            return restTemplateHelper.exchangeSafely(url, httpMethod, httpEntity, responseType, param);
        } catch (OpenApiException e) {
            if (e.getCode() == HttpStatus.UNAUTHORIZED.value()) {
                // 可能是token过期，发起重试
                LOG.info("重试请求：{}", url);
                ClientTokenHolder.refresh();
                return restTemplateHelper.exchangeSafely(url, httpMethod, httpEntity, responseType, param);
            }
            throw e;
        }
    }

    <T> T remoteExchangeCacheable(String url, HttpMethod httpMethod, HttpEntity<?> httpEntity, TypeReference<T> responseType,
                         Object... param) {
        // url + method + param作为key
        StringJoiner joiner = new StringJoiner("$@$");
        joiner.add(httpMethod.name());
        joiner.add(url);
        for (Object o : param) {
            if (o != null) {
                joiner.add(o.toString());
            }
        }

        String cacheKey = joiner.toString();
        return (T) EXCHANGE_CACHE.get(cacheKey, k -> remoteExchange(url, httpMethod, httpEntity, responseType, param));
    }

    void clearCache() {
        EXCHANGE_CACHE.invalidateAll();
    }

    /**
     * 获取创建人信息
     *
     * @return 创建人信息
     */
    UpdaterInfoDTO createUpdaterInfo() {
        UpdaterInfoDTO updaterInfoDTO = new UpdaterInfoDTO();

        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        if (authentication != null) {
            updaterInfoDTO.setUpdater(authentication.getName());
        }

        var request = this.currentRequest();
        if (request != null) {
            updaterInfoDTO.setIp(getClientIp(request));
            updaterInfoDTO.setBrowserAgent(request.getHeader(HttpHeaders.USER_AGENT));
        }

        return updaterInfoDTO;
    }

    HttpServletRequest currentRequest() {
        RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
        if (requestAttributes == null) {
            return null;
        }

        return ((ServletRequestAttributes) requestAttributes).getRequest();
    }

    /**
     * 获取客户端IP
     *
     * @param request request
     * @return IP
     */
    String getClientIp(HttpServletRequest request) {
        List<String> headers = Arrays.asList("X-Forwarded-For", "X-Real-IP", "Proxy-Client-IP", "WL-Proxy-Client-IP", "HTTP_CLIENT_IP", "HTTP_X_FORWARDED_FOR");
        for (String header : headers) {
            var headerValue = request.getHeader(header);
            if (isUnknown(headerValue)) {
                return getFirstValue(headerValue, ",");
            }
        }
        return request.getRemoteAddr();
    }

    private String getFirstValue(String value, String separator) {
        if (!StringUtils.hasText(value)) {
            return value;
        }

        for (String s : value.split(separator)) {
            var temp = s.trim();
            if (isUnknown(temp)) {
                continue;
            }
            return temp;
        }
        return value;
    }

    private boolean isUnknown(String ip) {
        return !StringUtils.hasText(ip) || "unknown".equalsIgnoreCase(ip);
    }

    static class AuthorizationInterceptor implements ClientHttpRequestInterceptor {

        @NonNull
        @Override
        public ClientHttpResponse intercept(HttpRequest request, @NonNull byte[] body, @NonNull ClientHttpRequestExecution execution) throws IOException {
            if (!CasUrlConstant.needToken(request.getURI())) {
                // 不需要token
                return execution.execute(request, body);
            }
            HttpHeaders headers = request.getHeaders();
            var token = ClientTokenHolder.getToken();
            if (token == null) {
                throw new IllegalStateException("Token获取失败");
            }

            // 设置token
            headers.add(HttpHeaders.AUTHORIZATION, token.getTokenType() + " " + token.getAccessToken());
            return execution.execute(request, body);
        }
    }
}
