package com.elitescloud.boot.dubbo.filter;

import com.elitescloud.boot.exception.BusinessException;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.common.extension.Activate;
import org.apache.dubbo.common.utils.ConfigUtils;
import org.apache.dubbo.rpc.*;
import org.apache.dubbo.validation.Validation;

import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import javax.validation.ValidationException;

import static org.apache.dubbo.common.constants.CommonConstants.CONSUMER;
import static org.apache.dubbo.common.constants.CommonConstants.PROVIDER;
import static org.apache.dubbo.common.constants.FilterConstants.VALIDATION_KEY;

/**
 * dubbo的自定义校验实现.
 *
 * @author Kaiser（wang shao）
 * @date 2021/5/18
 * @see org.apache.dubbo.validation.filter.ValidationFilter
 */
@Activate(group = {CONSUMER, PROVIDER}, value = VALIDATION_KEY, order = 9999)
@Slf4j
public class DubboValidationFilter implements Filter {

    private Validation validation;

    public void setValidation(Validation validation) {
        this.validation = validation;
    }

    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException {
        if (validation != null && !invocation.getMethodName().startsWith("$")
                && ConfigUtils.isNotEmpty(invoker.getUrl().getMethodParameter(invocation.getMethodName(), VALIDATION_KEY))) {
            try {
                var validator = validation.getValidator(invoker.getUrl());
                if (validator != null) {
                    validator.validate(invocation.getMethodName(), invocation.getParameterTypes(), invocation.getArguments());
                }
            } catch (RpcException e) {
                log.error("调用远程接口失败", e);
                throw new BusinessException("服务器繁忙，请稍后再试", e);
            } catch (ValidationException e) {
                // only use exception's message to avoid potential serialization issue
                log.error("dubbo异常：", e);
                if (e instanceof ConstraintViolationException) {
                    String msg = ((ConstraintViolationException) e).getConstraintViolations().toArray(ConstraintViolation[]::new)[0].getMessage();
                    return AsyncRpcResult.newDefaultAsyncResult(new ValidationException(msg), invocation);
                }
                return AsyncRpcResult.newDefaultAsyncResult(new ValidationException("调用远程接口失败，参数校验不通过"), invocation);
            } catch (Exception t) {
                log.error("dubbo异常：", t);
                return AsyncRpcResult.newDefaultAsyncResult(t, invocation);
            }
        }
        return invoker.invoke(invocation);
    }
}
