package com.elitescloud.boot.dubbo.util;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import com.alibaba.nacos.api.exception.NacosException;
import com.alibaba.nacos.api.naming.pojo.Instance;
import com.elitescloud.boot.SpringContextHolder;
import lombok.extern.log4j.Log4j2;
import org.apache.dubbo.common.constants.CommonConstants;
import org.apache.dubbo.common.constants.RegistryConstants;
import org.apache.dubbo.config.RegistryConfig;
import org.apache.dubbo.config.bootstrap.DubboBootstrap;
import org.apache.dubbo.config.bootstrap.builders.ReferenceBuilder;
import org.apache.dubbo.registry.Registry;
import org.apache.dubbo.registry.nacos.NacosNamingServiceWrapper;
import org.apache.dubbo.registry.nacos.NacosRegistry;
import org.apache.dubbo.registry.nacos.util.NacosNamingServiceUtils;
import org.apache.dubbo.registry.support.RegistryManager;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * dubbo常用工具.
 *
 * @author Kaiser（wang shao）
 * @date 2022/3/24
 */
@Log4j2
public abstract class DubboNacosHolder {

    /**
     * 获取注册配置
     *
     * @return
     */
    public static List<RegistryConfig> getRegistryConfig() {
        return new ArrayList<>(DubboBootstrap.getInstance().getConfigManager().getRegistries());
    }

    /**
     * 获取服务注册（应用级）
     *
     * @return 注册实例
     */
    public static List<Registry> getRegistries() {
        return new ArrayList<>(RegistryManager.getInstance(DubboBootstrap.getInstance().getApplicationModel())
                .getRegistries());
    }

    /**
     * 获取属于nacos的服务注册
     *
     * @return nacos注册
     */
    public static NacosRegistry getRegistryOfNacos() {
        var registries = getRegistries();
        if (registries.isEmpty()) {
            return null;
        }

        for (Registry registry : registries) {
            if (registry instanceof NacosRegistry) {
                return (NacosRegistry) registry;
            }
        }
        return null;
    }

    /**
     * 获取dubbo封装的nacosNamingService
     *
     * @return NacosNamingServiceWrapper
     */
    public static NacosNamingServiceWrapper getNacosNamingServiceWrapper() {
        var registry = getRegistryOfNacos();
        if (registry == null) {
            log.warn("未获取到NacosRegistry");
            return null;
        }

        var registryUrl = registry.getUrl();
        return NacosNamingServiceUtils.createNamingService(registryUrl);
    }

    /**
     * 构建服务名称
     *
     * @param category         服务类型 {@link RegistryConstants#PROVIDERS_CATEGORY}
     * @param serviceInterface 服务接口
     * @param version          版本号
     * @return 服务名称
     */
    public static String buildServiceName(String category, Class<?> serviceInterface, String version) {
        return Assert.notBlank(category) + ":" + serviceInterface.getName() + ":" +
                CharSequenceUtil.blankToDefault(version, "") + ":";
    }

    /**
     * 查询nacos服务实例
     *
     * @param serviceName 服务名称
     * @param group       服务分组
     * @return 服务实例
     * @throws NacosException 异常
     */
    public static List<Instance> queryNacosInstance(String serviceName, String group) throws NacosException {
        if (CharSequenceUtil.isBlank(group)) {
            group = SpringContextHolder.getProperty("dubbo.registry.group", "DEFAULT_GROUP");
        }

        var nacosNamingServiceWrapper = getNacosNamingServiceWrapper();
        if (nacosNamingServiceWrapper == null) {
            log.warn("未获取到NacosNamingServiceWrapper");
            return Collections.emptyList();
        }

        return nacosNamingServiceWrapper.getAllInstances(serviceName, group);
    }

    /**
     * 查询nacos服务实例
     * <p>
     * 根据应用名称分组
     *
     * @param serviceName 服务名称
     * @param group       服务分组
     * @return 服务实例
     * @throws NacosException 异常
     */
    public static Map<String, List<Instance>> queryNacosInstanceForApplication(String serviceName, String group) throws NacosException {
        var instances = queryNacosInstance(serviceName, group);
        if (instances.isEmpty()) {
            return Collections.emptyMap();
        }

        return instances.stream()
                .collect(Collectors.groupingBy(t -> t.getMetadata().getOrDefault(CommonConstants.APPLICATION_KEY, "unknown")));
    }

    /**
     * 获取特定的服务实例
     *
     * @param serviceInterface 服务接口
     * @param version          服务版本号
     * @param instance         特定的nacos服务实例
     * @param <T>              服务接口类型
     * @return 服务实例
     */
    public static <T> T getInstanceOfSpecial(Class<T> serviceInterface, String version, Instance instance) {
        return getInstanceOfSpecial(serviceInterface, version, instance, null);
    }

    /**
     * 获取特定的服务实例
     *
     * @param serviceInterface 服务接口
     * @param version          服务版本号
     * @param instance         特定的nacos服务实例
     * @param <T>              服务接口类型
     * @return 服务实例
     */
    public static <T> T getInstanceOfSpecial(Class<T> serviceInterface, String version, Instance instance, Integer timeout) {
        ReferenceBuilder<T> referenceBuilder = ReferenceBuilder.<T>newBuilder()
                .interfaceClass(serviceInterface)
                .version(version)
                .timeout(timeout)
                .url("dubbo://" + instance.getIp() + ":" + instance.getPort());
        if (timeout != null) {
            referenceBuilder.timeout(timeout);
        }
        return referenceBuilder.build().get();
    }
}
