package com.elitescloud.boot.flyway;

import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.flyway.common.FlywayBuilder;
import com.elitescloud.boot.flyway.common.FlywayHelper;
import com.elitescloud.cloudt.context.util.database.DatasourceLoadUtil;
import lombok.extern.log4j.Log4j2;
import org.flywaydb.core.Flyway;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.jdbc.DataSourceAutoConfiguration;
import org.springframework.boot.autoconfigure.jdbc.JdbcTemplateAutoConfiguration;
import org.springframework.boot.autoconfigure.orm.jpa.HibernateJpaAutoConfiguration;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.core.env.Environment;
import org.springframework.core.task.TaskExecutor;

/**
 * flyway相关自动化配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/3/23
 */
@ConditionalOnProperty(prefix = FlywayCloudtProperties.CONFIG_PREFIX, name = "enabled", havingValue = "true")
@EnableConfigurationProperties(FlywayCloudtProperties.class)
@ConditionalOnClass(Flyway.class)
@AutoConfigureAfter({DataSourceAutoConfiguration.class, JdbcTemplateAutoConfiguration.class,
        HibernateJpaAutoConfiguration.class})
@Import(CloudtFlywayAutoConfiguration.FlywayAutoMigrate.class)
@Log4j2
public class CloudtFlywayAutoConfiguration {

    private final FlywayCloudtProperties properties;

    public CloudtFlywayAutoConfiguration(FlywayCloudtProperties properties) {
        this.properties = properties;
    }

    @Bean
    @ConditionalOnMissingBean
    public FlywayBuilder flywayBuilder(Environment environment) {
        var dataSource = DatasourceLoadUtil.loadForHikari(environment);
        Assert.notNull(dataSource, "初始化flyway失败，未加载到有效数据源");
        return new FlywayBuilder(dataSource, properties);
    }

    @Bean
    public FlywayHelper flywayHolder() {
        return new FlywayHelper(properties);
    }

    @Order(Ordered.HIGHEST_PRECEDENCE)
    static class FlywayAutoMigrate implements ApplicationRunner {

        @Autowired
        private FlywayCloudtProperties flywayCloudtProperties;
        @Autowired
        private FlywayBuilder flywayBuilder;
        @Autowired
        private FlywayHelper flywayHelper;
        @Autowired
        private TaskExecutor taskExecutor;

        @Override
        public void run(ApplicationArguments args) throws Exception {
            taskExecutor.execute(() -> {
                log.info("检查【系统】数据库脚本...");
                checkSysScript();
                log.info("检查【系统】数据库脚本结束");
            });
        }

        private void checkSysScript() {
            if (flywayCloudtProperties.getAutoMigrateAfterStart().isNegative() || flywayCloudtProperties.getAutoMigrateAfterStart().isZero()) {
                log.info("检查【系统】数据库脚本忽略");
                return;
            }

            flywayHelper.migrate(flywayBuilder::createFlywayForSys);
        }
    }
}
